package RegAlloc.UCB;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import Digraph.DiNode;
import Graph.Node;
import Graph.SimpleGraph;
import Graph.SimpleNodeFc;
import RegAlloc.AllocationData;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.MemoryAccessInstruction;
import RegAlloc.MemoryAccessRegister;
import RegAlloc.PseudoReg;
import RegAlloc.RegAllocator;
import RegAlloc.Register;
import RegAlloc.ChordalAllocation.GreedyColoringAlgorithm;
import RegAlloc.ChordalAllocation.PostNodeCoalescer;
import RegAlloc.KrParser.InKr;
import RegAlloc.KrParser.OutKr;
import RegAlloc.RalfTools.DFA;

/**
 * This class performs register allocation via coloring of chordal graphs.
 * In order to perform spills, it builds the list of maximal cliques of the
 * graph and removes registers until the interference graph can be colored with
 * the given number of colors. The name of the class stands for the spilling
 * technique been used: Destruction of Maximal Cliques.
 */
public class UCAllocator implements RegAllocator {

    private PostNodeCoalescer pnc = null;

	private int highestColor = 0;

	private boolean isKColorable = true;

	private boolean isChordal = true;

	private HashMap<MemoryAccessRegister, Register> spilledRegisterAlias = new HashMap<MemoryAccessRegister, Register>();

	/**
	 * This set contains all the registers removed in the spilling process.
	 */
	private HashSet<Node<Register>> spilledNodes = new HashSet<Node<Register>>();

	/**
	 * This variable contains all the registers spilled in the last iteration
	 * of the spilling algorithm.
	 */
	private HashSet<Register> lastSpills = new HashSet<Register>();

	/**
	 * This variable is redundant, because every register in this set can also
	 * be found in the spilledNodes set; however, it is used due to eficiency
	 * reasons. Any register in this variable can be accessed in O(1).
	 */
	private HashSet<Register> spilledRegisters = new HashSet<Register>();

	private AllocationInstance ai = null;

	/**
	 */
	public void performRegAllocation() {
		int numColors = this.ai.rF.getNumMachineRegisters();
		ArrayList<MachineReg> regSet = null;
		do {
			this.isKColorable = true;
			this.updateControlFlowGraph();
			this.cleanControlFlowGraph();
			DFA.performLivenessAnalysis(this.ai, "0");
			this.buildInterferenceGraph();
			ListIterator<Node<Register>> tempList = this.getNodeOrdering();
			regSet = this.applyColoring(tempList, numColors);
			this.allocatePseudos(regSet);
		} while (!this.isKColorable);
		this.coalesceRegisters();
		this.allocatePseudos(regSet);
	}

	/**
	 * This method gets a table that maps registers to integer values and
	 * returns an ordered list where the ordering criteria is given by the
	 * integer values associated to the registers.
	 * @param uc the mapping of registers to integers.
	 * @return the ordered list of registers.
	 */
	private List<Register> orderRegisters(HashMap<Register, Integer> uc) {
		ArrayList<Register> list = new ArrayList<Register>(uc.size());
		for(Register reg : uc.keySet()) {
			list.add(reg);
		}
		for(int i = 0; i < list.size() - 1; i++) {
			for(int j = i + 1; j < list.size(); j++) {
				Register regI = list.get(i);
				Register regJ = list.get(j);
				Integer keyI = uc.get(regI);
				Integer keyJ = uc.get(regJ);
				if(keyI.intValue() < keyJ.intValue()) {
					list.set(j, regI);
					list.set(i, regJ);
				}
			}
		}
		return list;
	}

	/**
	 * This method order the nodes in the interference graph according to the
	 * usage frequency of each register. The usage frequency of a register is
	 * given by: sum, for all instruction i, freq(i)*(1, if reg is in use(i)
	 * or def(i)).
	 * @return the list of nodes, ordered in decresing order of usage
	 * frequency.
	 */
	private ListIterator<Node<Register>> getNodeOrdering() {
		Instruction i = this.ai.iF.createInstruction("0");
		Digraph.DiNode<Instruction> n = this.ai.cfg.getNode(i);
		UseCountCalculator ucc = new UseCountCalculator(this.ai);
		ucc.traverse(n);
		HashMap<Register, Integer> uc = ucc.getUseCount();
		List<Register> regList = this.orderRegisters(uc);
		ArrayList<Node<Register>> nodeList = new ArrayList<Node<Register>>();
		for(int j = 0; j < regList.size(); j++) {
			Node<Register> node = this.ai.ig.getNode(regList.get(j));
			nodeList.add(j, node);
		}
		return nodeList.listIterator();
	}

	private void cleanControlFlowGraph() {
		// reset the list of recent spills:
		this.lastSpills = new HashSet<Register>();
		// clean the control flow graph, eg.: remove the in and out set of all
		// the instruction:
		DFA.cleanControlFlowGraph(this.ai, "0");
	}

	/**
	 * This method updates the control flow graph so that it can accommodate
	 * spilled registers. Each spilled register will originate a new
	 * instruction.
	 */
	private void updateControlFlowGraph() {
		for (Register reg : this.lastSpills) {
			// place stores:
			Collection<Instruction> defInsts = reg.getDefInstructions();
			for (Instruction inst : defInsts) {
				// remove reg from inst
				inst.removeDef(reg);
				// creates a new instruction, where the new register will be inserted.
				MemoryAccessInstruction inst_store = this.ai.iF
						.createMemoryAccessInstruction(inst.getName());
				// create reg_store as a surrogate to the evicted register.
				MemoryAccessRegister reg_store = this.ai.rF
						.createMemoryAccessRegister(reg);
				inst.addDef(reg_store);
				reg_store.addDef(inst);
				inst_store.addUse(reg_store);
				reg_store.addUse(inst_store);
				// place the new instruction in the control flow graph:
				DiNode<Instruction> node = ai.cfg.getNode(inst);
				Collection<DiNode<Instruction>> succs = node.succs();
				Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
				for (DiNode<Instruction> succ : succs) {
					toBeRemoved.add(succ);
					succ.removePred(node);
					ai.cfg.connect(inst_store, succ.getData());
				}
				for (DiNode<Instruction> succ : toBeRemoved)
					// avoid concurrent ex
					node.removeSucc(succ);
				ai.cfg.connect(inst, inst_store);
				// make the alias: references to reg_store must be directed to reg
				this.spilledRegisterAlias.put(reg_store, reg);
			}

			// place loads:
			Collection<Instruction> useInsts = reg.getUseInstructions();
			for (Instruction inst : useInsts) {
				// remove reg from inst
				inst.removeUse(reg);
				// creates a new instruction, where the new register will be inserted.
				MemoryAccessInstruction inst_load = this.ai.iF
						.createMemoryAccessInstruction(inst.getName());
				// create reg_load as a surrogate to the loaded register.
				MemoryAccessRegister reg_load = this.ai.rF
						.createMemoryAccessRegister(reg);
				inst.addUse(reg_load);
				reg_load.addUse(inst);
				inst_load.addDef(reg_load);
				reg_load.addDef(inst_load);
				// place the new instruction in the control flow graph:
				DiNode<Instruction> node = ai.cfg.getNode(inst);
				Collection<DiNode<Instruction>> preds = node.preds();
				Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
				for (DiNode<Instruction> pred : preds) {
					toBeRemoved.add(pred);
					pred.removeSucc(node);
					ai.cfg.connect(pred.getData(), inst_load);
				}
				for (DiNode<Instruction> pred : toBeRemoved)
					// avoid concurrent ex
					node.removePred(pred);
				ai.cfg.connect(inst_load, inst);
				// make the alias: references to reg_load must be directed to reg
				this.spilledRegisterAlias.put(reg_load, reg);
			}
		}
	}

	/**
	 * In order to perform the spilling of the least used color, the interference
	 * graph is colored with many colors. The excess will be pruned by the
	 * spilling phase.
	 */
	private ArrayList<MachineReg> applyColoring(
			ListIterator<Node<Register>> regs, int numColors) {
		this.resetList(regs);

		// get the machine registers and color them
		ArrayList<MachineReg> mRegs = ai.rF.getMachineRegisters();
		for (int i = 0; i < mRegs.size(); i++)
			mRegs.get(i).setColor(i);

		// collor the pseudo registers.
		GreedyColoringAlgorithm gca = new GreedyColoringAlgorithm(numColors);
		this.highestColor = gca.color(regs);

		return mRegs;
	}

	/**
	 * This method points the list iterator to the first position in the list.
	 */
	public void resetList(ListIterator l) {
		while (l.hasPrevious()) {
			l.previous();
		}
	}

	/**
	 * This method returns the number of colors used to color this graph. If the
	 * graph is chordal, this number will be the size of the biggest clique, and
	 * the graph will be chordal.
	 * @return an integer value.
	 */
	public int getNumberOfColors() {
		return this.highestColor + 1;
	}

	/**
	 * Each color is associated to a machine register. This method assigns to
	 * each pseudo the machine register that is associated to the node's color.
	 */
	private void allocatePseudos(ArrayList<MachineReg> regSet) {
		ArrayList<PseudoReg> regs = ai.rF.getPseudoRegisters();
		for (Register reg : regs) {
			if (reg.getColor() < 0) {
				if (!this.spilledRegisters.contains(reg)) {
					Node<Register> node = this.ai.ig.getNode(reg);
					this.desconnectNode(node);
				}
			} else if (reg.getColor() < regSet.size()) {
				reg.setAllocationDestiny(regSet.get(reg.getColor()));
			}
		}
	}

	private void desconnectNode(Node<Register> n) {
		System.err.println("Spilling " + n.getData().getName());
		Collection<Node<Register>> j = n.neighbors();
		for (Node<Register> succ : j)
			succ.unlink(n);
		this.spilledNodes.add(n);
		this.spilledRegisters.add(n.getData());
		this.lastSpills.add(n.getData());
		this.isKColorable = false;
	}


	/**
	 * This method attempts to coalesce the pair of registers related by a move
	 * instruction. The coalescing of two registers, t1 and t2 is performed by
	 * greedy coloring. Greedy coloring is applied to the union of neighbors of t1
	 * and t2. Notice that the original colors of t1 or t2 may be used in the
	 * greedy coloring.
	 */
	public void coalesceRegisters() {
		this.pnc.coalesce(this.ai);
	}

	public Collection<Register> getAllocatedRegisters() {
		Collection<Node<Register>> c = this.ai.ig.getNodes();
		LinkedList<Register> l = new LinkedList<Register>();
		for (Node<Register> n : c) {
			if (!this.spilledNodes.contains(n)) {
				Register reg = n.getData();
				l.add(reg);
			}
		}
		java.util.Set<Register> colKeys = this.pnc.getCoalescedMapping()
				.keySet();
		for (Register reg : colKeys) {
			l.add(reg);
		}
		return l;
	}

	public Collection<Register> getSpilledRegisters() {
		LinkedList<Register> l = new LinkedList<Register>();
		for (Node<Register> n : this.spilledNodes) {
			Register reg = n.getData();
			l.add(reg);
		}
		return l;
	}

	public int numMachineRegisters() {
		return this.ai.rF.getNumMachineRegisters();
	}

	/**
	 * This method builds the interference graph from the liveness information
	 * stored in the control flow graph.
	 */
	private void buildInterferenceGraph() {
		this.ai.ig = new SimpleGraph<Register>();
		this.ai.ig.setNodeFactory(new SimpleNodeFc<Register>());

		// Add all the pseudo registers to the graph.
		ArrayList<PseudoReg> pseudoSet = this.ai.rF.getPseudoRegisters();
		for (PseudoReg pseudo : pseudoSet)
			if (!this.spilledRegisters.contains(pseudo))
				this.ai.ig.newNode(pseudo);

		// set the interferences between simultaneously alive registers
		Collection<DiNode<Instruction>> insts = this.ai.cfg.getNodes();
		for (DiNode<Instruction> nInst : insts) {
			ArrayList<Register> ll = nInst.getData().getInSet();
			for (int j = 0; j < ll.size(); j++) {
				Register r1 = ll.get(j);
				for (int k = j + 1; k < ll.size(); k++) {
					Register r2 = ll.get(k);
					this.ai.ig.connect(r1, r2);
				}
			}
		}

		// make the machine registers pairwise interfering
		ArrayList<MachineReg> regSet = this.ai.rF.getMachineRegisters();
		for (int i = 0; i < regSet.size() - 1; i++)
			for (int j = i + 1; j < regSet.size(); j++)
				this.ai.ig.connect(regSet.get(i), regSet.get(j));

		// reset the cardinality of all the nodes
	}

	public void setAllocationParameters(AllocationInstance ai) {
		this.ai = ai;
		this.pnc = new PostNodeCoalescer();
	}

	public AllocationData getAllocationData() {
		int numRegisters = this.ai.ig.size() + this.spilledNodes.size();
		int numColors = this.highestColor + 1;
		int numSpills = this.spilledNodes.size();
		int numMoves = this.pnc.getNumMoveInstructions();
		int numCoalescings = this.pnc.getNumCoalescedNodes();
		boolean isChordal = this.isChordal;
		return new AllocationData(numRegisters, numColors, numSpills, numMoves,
				numCoalescings, isChordal);
	}

	/**
	 * This method prints the numbers colected from the register allocation of the
	 * given graph. The output format is given by: title, chordability, total number
	 * of registers, number of colors used, number of spills, number of register
	 * coalescing.
	 * @param title the title of the graph
	 * @param out the output chanel where the information should be printed.
	 */
	public void printStatistics(String title, PrintStream out) {
		int numberOfRegisters = this.ai.ig.size() + this.spilledNodes.size();

		out.print(title + "\t\t " + this.isChordal + "\t\t "
				+ numberOfRegisters + "\t\t ");
		out.print((this.highestColor + 1) + "\t\t " + this.spilledNodes.size()
				+ "\t\t ");
		out.println(this.pnc.getNumCoalescedNodes() + "\t\t "
				+ this.pnc.getNumMoveInstructions());
	}

	public static void main(String a[]) throws IOException {
		if (a.length < 1) {
			System.err.println("java TestAlg in.dat");
			System.exit(1);
		} else {
			System.err.println("Running the usage based allocator...");
			AllocationInstance ai = InKr.read(a[0]);
			UCAllocator alloc = new UCAllocator();
			alloc.setAllocationParameters(ai);
			alloc.performRegAllocation();
//			DFA.drawControlFlowGraph(ai, "0", "fern.dot", true);
//			GraphvizGen gg = new GraphvizGen(ai, alloc);
//			gg.toGraphviz("result.dot", "XXXXX");
			OutKr ok = new OutKr(alloc, ai);
			ok.write();
			System.err.println("Done.");
		}
	}

}