package RegAlloc.SSA;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Stack;

import Digraph.DiNode;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.Register;

/**
 * This class renames variables, such that each variable has only one
 * definition. Notice, however, that it does not rename machine registers.
 */
public class VariableRenaming {

    private HashMap<Register, Stack<Register>> stacks = null;

    private HashMap<Register, Register> alias = null;

    private AllocationInstance ai = null;

    private DominanceTreeBuilder dtb = null;

    public VariableRenaming(AllocationInstance ai, DominanceTreeBuilder dtb) {
        this.dtb = dtb;
        this.ai = ai;
        this.stacks = new HashMap<Register, Stack<Register>>();
        this.alias = new HashMap<Register, Register>();
        for(Register a : this.ai.rF.getRegisters()) {
            Stack<Register> stack = new Stack<Register>();
            stack.push(a);
            this.alias.put(a, a);
            this.stacks.put(a, stack);
        }
    }

    public void rename(Instruction s) {
        if( ! (s instanceof PhiInstruction) ) {
            this.handleUses(s);
        }

        // handle definitions also in phi functions
        this.handleDefinitions(s);

        DiNode<Instruction> cfgNode = this.ai.cfg.getNode(s);
        for( DiNode<Instruction> succ : cfgNode.succs() ) {
            if( succ.getData() instanceof PhiInstruction ) {
                PhiInstruction pInst = (PhiInstruction)succ.getData();
                for(PhiFunction phi : pInst.getPhiFunctions()) {
                    Register param = this.alias.get(phi.getDef());
                    if( !(param instanceof MachineReg) ) {
                        Register newVar = this.stacks.get(param).peek();
                        phi.updateParam(s, newVar);
                    }
                }
            }
        }

        DiNode<Instruction> dtNode = this.dtb.getDominanceTree().getNode(s);
        for( DiNode<Instruction> x : dtNode.succs() ) {
            this.rename(x.getData());
        }

        for(Register a : s.getDefSet()) {
            if( ! (a instanceof MachineReg) ) {
                this.stacks.get(alias.get(a)).pop();
            }
        }

    }

    private void handleUses(Instruction s) {
        LinkedList<Register> addBag = new LinkedList<Register>();
        LinkedList<Register> delBag = new LinkedList<Register>();
        for( Register x : s.getUseSet() ) {
            if( ! (x instanceof MachineReg) ) {
                Register i = this.stacks.get(x).peek();
                addBag.add(i);
                delBag.add(x);
            }
        }
        for(Register reg : delBag) {
            s.removeUse(reg);
        }
        for(Register reg : addBag) {
            s.addUse(reg);
        }
    }

    private void handleDefinitions(Instruction s) {
        LinkedList<Register> addBag = new LinkedList<Register>();
        LinkedList<Register> delBag = new LinkedList<Register>();
        for(Register a : s.getDefSet()) {
            if( ! (a instanceof MachineReg) ) {
                PhiRegister pReg = this.ai.rF.createPhiRegister(a);
                this.stacks.get(a).push(pReg);
                this.alias.put(pReg, a);
                if( s instanceof PhiInstruction ) {
                    PhiInstruction pInst = (PhiInstruction)s;
                    pInst.updateDef(pReg, a);
                } else {
                    addBag.add(pReg);
                    delBag.add(a);
                }
            }
        }
        for(Register reg : delBag) {
            s.removeDef(reg);
        }
        for(Register reg : addBag) {
            s.addDef(reg);
        }
    }

}
