package RegAlloc.SSA;

import Digraph.DiNode;
import Digraph.Digraph;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.KrParser.InKr;
import RegAlloc.RalfTools.DFA;

public class SSA {

    public static void main(String argv[]) {
        if(argv.length < 1) {
            System.err.println("Sintaxe: java5 SSA file.dat");
            System.exit(1);
        }

        AllocationInstance ai = InKr.read(argv[0]);
        SSA.convertToSSA(ai);
        SSA.drawCFG(ai);
    }


    public static void convertToSSA(AllocationInstance ai) {
        // get the first instruction/node of the CFG:
        Instruction startInst = ai.iF.createInstruction("0"); 
        DiNode<Instruction> startNode = ai.cfg.getNode(startInst);

        // builds the dominance tree:
        DominanceTreeBuilder dtb = new DominanceTreeBuilder(ai);
        dtb.traverse(startNode);
        dtb.getDominanceTree();

        // buids the dominance frontier of each node:
        DominanceFrontierBuilder dfb = new DominanceFrontierBuilder(ai, dtb);
        dfb.getDF(startNode.getData());

        // distribute phi functions accross the CFG:
        PhiSower ps = new PhiSower(ai, dfb, dtb);
        ps.placePhiFunctions();

        // rename the variables:
        VariableRenaming vr = new VariableRenaming(ai, dtb);
        vr.rename(startInst);

        // split the phi functions:
        PhiSplitter psp = new PhiSplitter(ai);
        psp.traverse(startNode);
    }


    public static void drawCFG(AllocationInstance ai) {
        DFA.cleanControlFlowGraph(ai, "0");
        DFA.performLivenessAnalysis(ai, "0");
        DFA.drawControlFlowGraph(ai, "0", false);
    }

    /**
     * It is recommended that, before the register allocation, the phi
     * functions be split among the different edges that created them. This
     * step makes easier to perform the liveness analysis.
     */
    public static void splitPhiFunctions(AllocationInstance ai) {
        Instruction startInst = ai.iF.createInstruction("0"); 
        DiNode<Instruction> startNode = ai.cfg.getNode(startInst);

        PhiSplitter psp = new PhiSplitter(ai);
        psp.traverse(startNode);
    }

    public static void renameVariables(AllocationInstance ai, DominanceTreeBuilder dtb) {
        Instruction startInst = ai.iF.createInstruction("0"); 
        VariableRenaming vr = new VariableRenaming(ai, dtb);
        vr.rename(startInst);
    }

    public static void placePhi(AllocationInstance ai, DominanceTreeBuilder dtb, DominanceFrontierBuilder dfb) {
        PhiSower ps = new PhiSower(ai, dfb, dtb);
        ps.placePhiFunctions();
    }


    public static DominanceFrontierBuilder buildDF(AllocationInstance ai, DominanceTreeBuilder dtb) {
        Instruction startInst = ai.iF.createInstruction("0"); 
        DiNode<Instruction> startNode = ai.cfg.getNode(startInst);

        DominanceFrontierBuilder dfb = new DominanceFrontierBuilder(ai, dtb);
        dfb.getDF(startNode.getData());
        return dfb;
    }

    public static DominanceTreeBuilder buildDT(AllocationInstance ai) {
        Instruction startInst = ai.iF.createInstruction("0"); 
        DiNode<Instruction> startNode = ai.cfg.getNode(startInst);

        DominanceTreeBuilder dtb = new DominanceTreeBuilder(ai);
        dtb.traverse(startNode);
        return dtb;
    }

    public static void drawGraph(Digraph<Instruction> graph, AllocationInstance ai) {
        Digraph<Instruction> aux = ai.cfg;
        ai.cfg = graph;
        DFA.drawControlFlowGraph(ai, "0", false);
        ai.cfg = aux;
    }

}
