package RegAlloc.SSA;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;

import utilities.ListOp;
import Digraph.DepthFirstSearch;
import Digraph.DiNode;
import Digraph.DiNodeFc;
import Digraph.Digraph;
import Digraph.SimpleDigraph;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;

public class DominanceTreeBuilder extends DepthFirstSearch<Instruction> {

    private Digraph<Instruction> dTree = null;

    private AllocationInstance ai = null;

    private HashMap<Instruction, Collection<Instruction>> dominators = null;

    private HashMap<Instruction, Collection<Instruction>> domains = null;

    private HashMap<Instruction, Boolean> isDone = null;

    private HashMap<Instruction, Instruction> iDoms = null;

    public DominanceTreeBuilder(AllocationInstance ai) {
        this.ai = ai;
        this.dominators = new HashMap<Instruction, Collection<Instruction>>();
        this.domains = new HashMap<Instruction, Collection<Instruction>>();
        this.isDone = new HashMap<Instruction, Boolean>();
        for(DiNode<Instruction> n : this.ai.cfg.getNodes()) {
            Instruction inst = n.getData();
            Collection<Instruction> doms = ListOp.cp(this.ai.iF.getInstructions());
            this.dominators.put(inst, doms);
            this.isDone.put(inst, new Boolean(false));
        }
    }

    public Collection<Instruction> getDomain(Instruction inst) {
        if( ! this.domains.containsKey(inst) ) {
            DomainCollector dc = new DomainCollector();
            DiNode<Instruction> n = this.dTree.getNode(inst);
            dc.traverse(n);
            Collection<Instruction> domain = dc.getDomain();
            domain.remove(inst);
            this.domains.put(inst, domain);
        }
        return this.domains.get(inst);
    }

    public void visit(DiNode<Instruction> n) {
        Instruction inst = n.getData();

        for(Instruction tempInst : this.ai.iF.getInstructions()) {
            this.isDone.put(tempInst, new Boolean(false));
        }

        Collection<Instruction> oldList = new HashSet<Instruction>();
        Collection<Instruction> newList = ListOp.cp(this.dominators.get(inst));

        do {

            oldList = ListOp.cp(newList);
            newList = this.getDominators(n);

        } while( ! ListOp.eq(oldList, newList) );

    }

    public Digraph<Instruction> getDominanceTree() {
        if(this.iDoms == null) {
            this.iDoms = new HashMap<Instruction, Instruction>();
            for(Instruction inst : this.dominators.keySet()) {
                if(!inst.getName().equals("0")) {
                    this.iDoms.put(inst, this.getIDom(inst));
                }
            }
        }
        if(dTree == null) {
            dTree = new SimpleDigraph<Instruction>();
            dTree.setNodeFactory(new DiNodeFc<Instruction>());
            for(Instruction inst : this.iDoms.keySet()) {
                dTree.connect(this.iDoms.get(inst), inst);
            }
        }
        return dTree;
    }


    public void insertBeforeDTInstruction(Instruction newInst, Instruction oldInst) {
        // place the new instruction in the control flow graph:
        DiNode<Instruction> node = this.dTree.getNode(oldInst);
        Collection<DiNode<Instruction>> preds = node.preds();
        Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
        for(DiNode<Instruction> pred : preds) {
            toBeRemoved.add(pred);
            pred.removeSucc(node);
            this.dTree.connect(pred.getData(), newInst);
        }
        for(DiNode<Instruction> pred : toBeRemoved) // avoid concurrent ex
            node.removePred(pred);
        this.dTree.connect(newInst, oldInst);
    }


    public Instruction getIDom(Instruction inst) {
        if(this.iDoms.containsKey(inst)) {
            return this.iDoms.get(inst);
        } else {
            Collection<Instruction> aux = ListOp.cp(this.dominators.get(inst));
            aux.remove(inst);
            Instruction iDom = ListOp.first(aux);
            for( Instruction dom : aux ) {
                if( this.dominates(iDom, dom)) {
                    iDom = dom;
                }
            }
            return iDom;
        }
    }

    public boolean dominates(Instruction a, Instruction b) {
        return this.dominators.get(b).contains(a);
    }

    public Collection<Instruction> getDominators(DiNode<Instruction> n) {
        Instruction inst = n.getData();

        if( ! this.isDone.get(inst).booleanValue() ) {

            this.isDone.put(inst, new Boolean(true));

            Collection<Instruction> domList = new HashSet<Instruction>();

            boolean isFirst = true;
            for(DiNode<Instruction> pred : n.preds()) {
                if(isFirst) {
                    domList = ListOp.cp(getDominators(pred));
                    isFirst = false;
                } else {
                    domList = ListOp.it(domList, getDominators(pred));
                }
            }

            domList.add(inst);

            this.dominators.put(inst, domList);
        }
        return this.dominators.get(inst);
    }


    public String toString() {
        String ans = "";
        for(Instruction inst : this.dominators.keySet()) {
            ans += inst.getName() + ": ";
            for(Instruction dom : this.dominators.get(inst)) {
                ans += dom.getName() + " ";
            }
            ans += "\n";
        }
        return ans;
    }

}
