package RegAlloc;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Set;

import RegAlloc.SSA.PhiRegister;

/**
 * The register factory is responsible for the creation of machine and pseudo
 * registers. The advantage of using an object factory is the ability of keeping
 * all the creation points in a single piece of code. This factory keeps all the
 * registers created in one of two lists: the list of machine registers, or the
 * list of pseudo-registers.
 */
public class RegisterFactory {

	private HashMap<String, MachineReg> machineRegs = new HashMap<String, MachineReg>();

	private HashMap<String, PseudoReg> pseudoRegs = new HashMap<String, PseudoReg>();

	private HashMap<Register, Register> memoryAccessTable = new HashMap<Register, Register>();

	private HashSet<MemoryAccessRegister> memoryAccessRegs = new HashSet<MemoryAccessRegister>();

	private LinkedList<String> callerSave = new LinkedList<String>();

	private LinkedList<PhiRegister> phiRegs = new LinkedList<PhiRegister>();

	/**
	 * This method generates a new machine register. If the register has been
	 * already created, then the old instance will be returned. Otherwise, a new
	 * object will be created.
	 * 
	 * @param id
	 *            the name of the new register.
	 * @return an object of the <CODE>MachineReg</CODE> type.
	 */
	public MachineReg createMachineRegister(String id) {
		if (this.machineRegs.containsKey(id))
			return this.machineRegs.get(id);
		MachineReg r = new MachineReg(id);
		this.machineRegs.put(id, r);
		return r;
	}

	/**
	 * This method ascribes the caller save status to the machine register
	 * represented by the string id.
	 */
	public void makeCallerSave(String id) {
		if (!this.machineRegs.containsKey(id))
			this.createMachineRegister(id);
		this.callerSave.add(id);
	}

	/**
	 * Memory access registers are pseudo registers created due to the spilling
	 * of nodes during register allocation. The method <CODE>getPseudoRegisters</CODE>
	 * will return the memory access registers in addition to the ordinary
	 * pseudo registers. There is an important distinction between this method
	 * and the other creation methods. While in the others a new register is
	 * created only if the given identifier has never been seem, here a new
	 * register is always created. Memory access have different names, even if
	 * they are created with the same identifier. A static counter from the
	 * Memory access class is used to differentiate them.
	 * 
	 * @param id
	 *            the name of the new register.
	 * @return an object of the <CODE>MachineReg</CODE> type.
	 */
	public MemoryAccessRegister createMemoryAccessRegister(Register origin) {
		MemoryAccessRegister r = new MemoryAccessRegister(origin);
		this.memoryAccessRegs.add(r);
		this.memoryAccessTable.put(r, origin);
		return r;
	}

	/**
	 * Phi registers are created when a program is transformed to the SSA form.
	 * Every time a variable is renamed, a phi register is created to represent
	 * the new live range.
	 */
	public PhiRegister createPhiRegister(Register origin) {
		PhiRegister r = new PhiRegister(origin);
		this.phiRegs.add(r);
		return r;
	}

	/**
	 * This method generates a new pseudo register. If the register has been
	 * already created, then the old instance will be returned. Otherwise, a new
	 * object will be created.
	 * 
	 * @param id
	 *            the name of the new register.
	 * @return an object of the <CODE>MachineReg</CODE> type.
	 */
	public PseudoReg createPseudoRegister(String id) {
		if (this.pseudoRegs.containsKey(id))
			return this.pseudoRegs.get(id);
		PseudoReg r = new PseudoReg(id);
		this.pseudoRegs.put(id, r);
		return r;
	}

	/**
	 * A memory access register is created as a surrugate to a temporary that
	 * has been sent to memory. Given a memory access register, this method
	 * returns the original register.
	 * 
	 * @param mar
	 *            a register.
	 * @return if the parameter is a memory access register, the original
	 *         register that the parameter represents will be returned,
	 *         otherwise null will be returned.
	 */
	public Register getOriginalRegister(Register mar) {
		return this.memoryAccessTable.get(mar);
	}

	/**
	 * This method returns an array containing the machine registers. The array
	 * is formed in a way that the caller saved registers are placed in its end,
	 * so that the ordering of registers also suits a greedy coloring algorithm.
	 */
	public ArrayList<MachineReg> getMachineRegisters() {
		ArrayList<MachineReg> list = new ArrayList<MachineReg>(machineRegs
				.size());
		ArrayList<String> non_caller_save = new ArrayList<String>();
		Set<String> regNames = this.machineRegs.keySet();
		for (String regName : regNames)
			if (!callerSave.contains(regName))
				non_caller_save.add(regName);
		for (int i = 0; i < non_caller_save.size(); i++)
			list.add(this.machineRegs.get(non_caller_save.get(i)));
		ListIterator<String> iter = this.callerSave.listIterator();
		while (iter.hasNext())
			list.add(this.machineRegs.get(iter.next()));
		return list;
	}

	/**
	 * Informs if a name is a valid identifier of a machine register
	 * @param regName a String object
	 * @return true if the parameter is the identifier of a machine register,
	 * and false otherwise.
	 */
	public boolean isValidMachineRegister(String regName) {
		return this.machineRegs.containsKey(regName);
	}

	/**
	 * Returns a list of the caller save registers.
	 */
	public ArrayList<MachineReg> getCallerSaveRegisters() {
		ArrayList<MachineReg> list = new ArrayList<MachineReg>(callerSave
				.size());
		ListIterator<String> iter = this.callerSave.listIterator();
		while (iter.hasNext())
			list.add(this.machineRegs.get(iter.next()));
		return list;
	}

	/**
	 * This method informs the number of machine registers available to solve
	 * the register allocation problem.
	 */
	public int getNumMachineRegisters() {
		return this.machineRegs.size();
	}

	public ArrayList<PseudoReg> getPseudoRegisters() {
		ArrayList<PseudoReg> list = new ArrayList<PseudoReg>(pseudoRegs.size());

		Collection<PseudoReg> prs = this.pseudoRegs.values();
		for (PseudoReg pr : prs)
			list.add(pr);

		for (PseudoReg pr : this.memoryAccessRegs)
			list.add(pr);

		for (PhiRegister phr : this.phiRegs)
			list.add(phr);

		return list;
	}

	/**
	 * Informs if a name is a valid identifier of a pseudo register. Notice
	 * that, even though memory access registers are considered pseudo
	 * registers, this method does not take into consideration the memory
	 * access registers.
	 * @param regName a String object
	 * @return true if the parameter is the identifier of a pseudo register,
	 * and false otherwise.
	 */
	public boolean isValidPseudoRegister(String regName) {
		return this.pseudoRegs.containsKey(regName);
	}

	/**
	 * This method returns an union of all the registers created by the factory.
	 */
	public LinkedList<Register> getRegisters() {
		LinkedList<Register> list = new LinkedList<Register>();
		Collection<MachineReg> mrs = this.machineRegs.values();
		for (MachineReg mr : mrs)
			list.add(mr);
		Collection<PseudoReg> prs = this.pseudoRegs.values();
		for (PseudoReg pr : prs)
			list.add(pr);
		for (MemoryAccessRegister mar : this.memoryAccessRegs)
			list.add(mar);
		return list;
	}

	public String toString() {
		String m = "Machine registers: ";
		Collection<MachineReg> mrs = this.machineRegs.values();
		for (MachineReg mr : mrs)
			m += mr.getName() + " ";

		String p = "Pseudo registers: ";
		Collection<PseudoReg> prs = this.pseudoRegs.values();
		for (PseudoReg pr : prs)
			p += pr.getName() + " ";

		String c = "Caller-save registers: ";
		for (String s : this.callerSave) {
			MachineReg mr = this.machineRegs.get(s);
			c += mr.getName() + " ";
		}

		return m + '\n' + p + '\n' + c;
	}

}
