package RegAlloc;

import java.util.Collection;
import java.util.LinkedList;

/**
 * This class represents a single registers in the allocation process. The
 * register contains references to the instructions where it is defined
 * and used. There are two types of registers in this implementation:
 * pre-colored registers and pseudo-registers. The pre-coloared registers
 * represent the machine registers available for allocation. The pseudo
 * registers represent temporary variables that the compiler generate
 * as part of its intermediate representation. In this implementation,
 * the pre-colored registers have names such as 'R0', 'R1', etc, whereas
 * the pseudos are represented only by numbers.
 */
public class Register extends CardNode {

    private int liveRangeSize = 1;

    /**
     * This parameter denotes the place that will be allocated to this register
     * after the register allocation phase. If this is a pre-colored register,
     * this place will be the register itself; otherwise, a pre-colored
     * register will be allocated to this pseudo.
     */
    private Register machineLoc = null;
    private String id = null;
    private LinkedList<Instruction> whereIsDef = null;
    private LinkedList<Instruction> whereIsUsed = null;

    public Register(String id) {
        this.id = id;
        this.whereIsDef = new LinkedList<Instruction>();
        this.whereIsUsed = new LinkedList<Instruction>();
        this.liveRangeSize = 1;
    }

    public void addDef(Instruction i) {
        whereIsDef.add(i);
    }

    public Collection<Instruction> getDefInstructions() {
        return this.whereIsDef;
    }

    public void addUse(Instruction i) {
        whereIsUsed.add(i);
    }

    public Collection<Instruction> getUseInstructions() {
        return this.whereIsUsed;
    }

    public String getName() {
        return id;
    }

    public double spillingFactor() {
        int useFrequency = 0;
        for(Instruction inst : whereIsDef)
            useFrequency += inst.getFrequency();
        for(Instruction inst : whereIsUsed)
            useFrequency += inst.getFrequency();

        // only one type cast is necessary, because double op int -> double
        return (double)this.liveRangeSize / useFrequency;
    }

    /**
     * This method is used to determine the spilling factor for the register.
     * It allows to record the size of the live range of the register during
     * the liveness analysis.
     */
    public void incLiveRange() {
        this.liveRangeSize++;
    }

    public void resetLiveRange() {
        this.liveRangeSize = 1;
    }

    /**
     * In some applications it is necessary to know the integer representation
     * of the register. For example, in Krishna's framework, the name of each
     * register is an integer number. Sometimes it is necessary to order the
     * registers by name.
     */
    public int getIntegerName() {
        return Integer.parseInt(id);
    }


    public String toString() {
        return this.getName();
    }

    public void setAllocationDestiny(Register r) {
        if( ( r != null ) && ! (r instanceof MachineReg) ) {
            System.err.println("Error: attempt to bind register " + this.getName() + " to non-machine register " + r.getName());
            System.exit(1);
        }
        this.machineLoc = r;
    }


    /**
     * Returns the place where this register will be allocated in the final
     * code. If this register is a pre-colored register, then the allocation
     * destiny will be its proper name, without the prefix. For instance,
     * 'R0' will return 0. If this is a pseudo-register, then the location
     * returned will be the register where it has been allocated.
     */
    public Register getAllocationDestiny() {
        return this.machineLoc;
    }


    /**
     * The caller saved registers are R0, R1, R2, R3 and R9. This concept
     * only applies to pseudo-registers. This method is always false for
     * pre-colored registers.
     */
    public boolean isStoredInCallerSaved() {
        return false;
    }

    public boolean spilled() {
        return this.machineLoc == null;
    }

    public boolean isPreColoredRegister() {
        return true;
    }

    /*
     * The following methods generate the output for Krishna's framework.
     */

    /**
     * This method prints, for each instruction inst that defines this
     * register, a string such as <CODE>inst pseudo color</CODE>.
     * @return an object of the <CODE>String</CODE> type.
     */
    public String defOutput() {
        return "";
    }


    /**
     * This method prints the binds between spilled pseudos and the registers
     * in which they are loaded. For example, if this is pseudo 4, and it will
     * be loaded in register 8 in instruction 54, the following output will be
     * given: <CODE>54 4 8</CODE>.
     */
    public String PsR() {
        return "";
    }

}
