package RegAlloc.RalfTools;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;

import utilities.ListOp;
import Graph.Node;
import RegAlloc.AllocationData;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.PseudoReg;
import RegAlloc.RegAllocator;
import RegAlloc.Register;

/**
 * This class prints the result of the register allocation as a graph in the
 * .dot format.
 */
public class GraphvizGen {

    private AllocationData ad;
    private AllocationInstance ai;
    private RegAllocator ra;

    public GraphvizGen(AllocationInstance ai, RegAllocator ra) {
        this.ad = ra.getAllocationData();
        this.ai = ai;
        this.ra = ra;
    }

    private void printChordalInformation(PrintWriter out) {
        if(this.ad.isChordal) {
            out.println(" * This is a chordal graph. It can be optimally colored with " + (this.ad.numColors) + " colors.");
            Collection<Register> regs = this.ra.getAllocatedRegisters();
            for(Register reg : regs) {
                out.println(" * " + reg);
            }
        } else {
            out.println(" * This is not a chordal graph. The proposed coloring uses " + (this.ad.numColors) + " colors.");
        }
    }


    private void printSpilledNodes(PrintWriter out) {
        Collection<Register> c = this.ra.getSpilledRegisters();
        if (c.size() > 0) {
            out.println(" * ------------------------------------------------");
            out.println(" * The spilled nodes are:");
            for(Register reg : c)
                out.println(" * " + reg.getName());
        }
    }


    private HashSet<Node<Register>> printRemainingGraph(PrintWriter out) {
        HashSet<Node<Register>> sMap = new HashSet<Node<Register>>();
        Collection<Node<Register>> i = this.ai.ig.getNodes();
        for(Node<Register> n : i) {
            sMap.add(n);
            Collection<Node<Register>> j = n.neighbors();
            for(Node<Register> succ : j) {
                if(!sMap.contains(succ)) {
                    String origin = n.getData().toString();
                    String destiny = succ.getData().toString();
                    out.println("\t\"" + origin + "\" -- \"" + destiny + "\";");
                }
            }
        }
        return sMap;
    }


    public void printInterferenceGraph(String fileName, String graphName) throws IOException {
        PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
        out.println("graph " + graphName + " {\n");
        out.println("node [\n");
        out.println("    style = \"filled\"");
        out.println("];\n");

        // determine the color of machine registers
        ArrayList<MachineReg> macRegs = this.ai.rF.getMachineRegisters();
        for(MachineReg mr : macRegs)
            out.println(mr.getName() + "[color=red]");

        // determine the color of pseudo registers
        ArrayList<PseudoReg> psRegs = this.ai.rF.getPseudoRegisters();
        for(PseudoReg pr : psRegs)
            out.println(pr.getName() + "[color=green]");

        // printing the interfering edges
        HashSet<Node<Register>> alreadyVisited = new HashSet<Node<Register>>();
        Collection<Node<Register>> nodes = this.ai.ig.getNodes();
        for(Node<Register> node : nodes) {
            alreadyVisited.add(node);
            Collection<Node<Register>> neighs = node.neighbors();
            for(Node<Register> neigh : neighs)
                if(alreadyVisited.contains(neigh) == false)
                    out.println(node.getData().getName() + " -- " + neigh.getData().getName());
        }

        // print the move related edges
        LinkedList<Instruction> insts = this.ai.iF.getMoveInstructions();
        for(Instruction inst : insts) {
            Register x = ListOp.first(inst.getDefSet());
            Register y = ListOp.first(inst.getUseSet());
            out.println(y.getName() + " -- " + x.getName() + "[style=dashed arrowhead=open]");
        }

        out.println("\n}");
        out.close();
    }


    /**
     * This method outputs the graph in the graphviz format. An example of such
     * file is:
     * <PRE>
     * graph graphName {
     *    "R8(0)" -- "R11(1)";
     *    "R8(0)" -- "R7(1)";
     *    "R11(1)" -- "R9(0)";
     * }
     * </PRE>
     * @param fileName the name of the file that will contain the output.
     * @param graphName the name of the graph.
     */
    public void toGraphviz(String fileName, String graphName) throws IOException {
        PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
//        this.mountCoalescingLists();
        out.println("/*");
        this.printChordalInformation(out);
        this.printSpilledNodes(out);
        out.println(" */\n");
        out.println("graph " + graphName + " {\n");
        this.printRemainingGraph(out);
        out.println("\n}");
        out.close();
    }

}
