package RegAlloc.RalfTools;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;

import utilities.ListOp;
import Digraph.DepthFirstSearch;
import Digraph.DiNode;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.MemoryAccessInstruction;
import RegAlloc.Register;
import RegAlloc.RegisterBinding;
import RegAlloc.KrParser.FordReader;

public class FordSower extends DepthFirstSearch<Instruction> {

	private BRegFactory brf = null;

	private FordReader fr = null;

	private AllocationInstance ai = null;

	public FordSower(FordReader fr, BRegFactory brf, AllocationInstance ai) {
		this.fr = fr;
		this.brf = brf;
		this.ai = ai;
	}

	public void visit(DiNode<Instruction> n) {
		Instruction inst = n.getData();

		if (!(inst instanceof MemoryAccessInstruction)) {
			this.updateDefs(inst);
			this.updateUses(inst);
			this.insertLoads(n);
			this.insertStores(n);
		}
	}

	private void insertStores(DiNode<Instruction> n) {
		Instruction inst = n.getData();
		Collection<RegisterBinding> stores = this.fr.readSpStore();
		for (RegisterBinding rb : stores) {
			if (rb.getInstruction().equals(inst)) {
				if (this.ai.iF.isJump(inst)) {
					System.err
							.println("Warning: attempt to place store after branch instruction: "
									+ inst);
				} else {
					// creates a new instruction, where the new register will be
					// inserted.
					MemoryAccessInstruction inst_store = this.ai.iF
							.createMemoryAccessInstruction(inst.getName());
					BoundReg reg_store = this.brf.createBoundRegister(rb
							.getPseudoReg(), rb.getMachineReg());
					inst_store.addUse(reg_store);
					// connect the new instruction in the control flow graph
					// remember: the old instruction has exactly one successor
					DiNode<Instruction> succ = ListOp.first(n.succs());
					n.removeSucc(succ);
					succ.removePred(n);
					this.ai.cfg.connect(inst_store, succ.getData());
					this.ai.cfg.connect(inst, inst_store);
				}
			}
		}
	}

	private void insertLoads(DiNode<Instruction> node) {
		Instruction inst = node.getData();

		Collection<RegisterBinding> loads = this.fr.readSpLoad();
		for (RegisterBinding rb : loads) {
			if (rb.getInstruction().equals(inst)) {
				// creates a new instruction, where the new register will be
				// inserted.
				MemoryAccessInstruction inst_load = this.ai.iF
						.createMemoryAccessInstruction(inst.getName());
				BoundReg reg_load = this.brf.createBoundRegister(rb
						.getPseudoReg(), rb.getMachineReg());
				inst_load.addDef(reg_load);
				// connect the new instruction in the control flow graph
				Collection<DiNode<Instruction>> preds = node.preds();
				Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
				for (DiNode<Instruction> pred : preds) {
					toBeRemoved.add(pred);
					pred.removeSucc(node);
					this.ai.cfg.connect(pred.getData(), inst_load);
				}
				for (DiNode<Instruction> pred : toBeRemoved)
					// avoid concurrent ex
					node.removePred(pred);
				this.ai.cfg.connect(inst_load, inst);
			}
		}
	}

	private void updateUses(Instruction inst) {
		Collection<Register> aux1 = new HashSet<Register>();
		Collection<BoundReg> aux2 = new HashSet<BoundReg>();

		Collection<RegisterBinding> psr = this.fr.readPsR();
		Collection<Register> uses = inst.getUseSet();
		for (Register reg : uses) {
			if (reg instanceof MachineReg) {
				MachineReg mr = (MachineReg) reg;
				BoundReg br = this.brf.createBoundRegister(mr, mr);
				aux1.add(reg);
				aux2.add(br);
			} else {
				for (RegisterBinding rb : psr) {
					if (rb.getPseudoReg().equals(reg)
							&& rb.getInstruction().equals(inst)) {
						BoundReg br = this.brf.createBoundRegister(rb
								.getPseudoReg(), rb.getMachineReg());
						aux1.add(reg);
						aux2.add(br);
					}
				}
			}
		}
		for (Register reg : aux1)
			inst.removeUse(reg);
		for (Register reg : inst.getUseSet()) {
			if (!(reg instanceof MachineReg)) {
				System.err.println("Error: instruction " + inst
						+ " has a pseudo not bound to a machine register: "
						+ reg);
				System.exit(1);
			}
		}
		for (BoundReg reg : aux2)
			inst.addUse(reg);
	}

	private void updateDefs(Instruction inst) {
		Collection<Register> aux1 = new HashSet<Register>();
		Collection<BoundReg> aux2 = new HashSet<BoundReg>();

		// fill up the definitions with register bindings
		Collection<RegisterBinding> xdf = this.fr.readXDef();
		Collection<Register> defs = inst.getDefSet();
		for (Register reg : defs) {
			if (reg instanceof MachineReg) {
				MachineReg mr = (MachineReg) reg;
				BoundReg br = this.brf.createBoundRegister(mr, mr);
				aux1.add(reg);
				aux2.add(br);
			} else {
				for (RegisterBinding rb : xdf) {
					if (rb.getPseudoReg().equals(reg)
							&& rb.getInstruction().equals(inst)) {
						BoundReg br = this.brf.createBoundRegister(rb
								.getPseudoReg(), rb.getMachineReg());
						aux1.add(reg);
						aux2.add(br);
					}
				}
			}
		}
		for (Register reg : aux1)
			inst.removeDef(reg);
		for (Register reg : inst.getDefSet()) {
			if (!(reg instanceof MachineReg)) {
				System.err.println("Error: instruction " + inst
						+ " has a pseudo not bound to a machine register: "
						+ reg);
				System.exit(1);
			}
		}
		for (BoundReg reg : aux2)
			inst.addDef(reg);

	}

}
