package RegAlloc.RalfTools;

import java.util.ArrayList;
import java.util.Collection;

import Digraph.DepthFirstSearch;
import Digraph.DiNode;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.Register;

/**
 * This class scans the control flow graph searching for bugs. There are four
 * types of bugs: <CODE>BAD_ALLOC</CODE>: an error in the set of register
 * allocation directives causes a temporary t to be defined in a register r1
 * that is different from the register r2 in which t is expected to be found.
 * <CODE>UNDEF_ALLOC</CODE>: there is a path p in the set S of states from
 * the initial instruction i0 to an instruction i where t is used, and t is not
 * defined along p. Such situation can be an error in the register allocator, or
 * it can be a deficiency of the high level programming language been compiled.
 * Non-strict languages, such as C, allow the use of undefined variables. This
 * is not possible in strict languages, such as Java. <CODE>INST_OVT</CODE>:
 * There is a path in S in which the definition of a temporary t1 is overwritten
 * by the definition of a temporary t2 while t1 is still alive. <CODE>CALL_OVT</CODE>:
 * A temporary t, stored in a caller save register, has its location overwritten
 * because it is alive across a function call.
 * 
 * @author fernando
 * 
 */
public class BugHunter extends DepthFirstSearch<Instruction> {

	private AllocationInstance ai = null;

	private boolean foundError = false;

	public BugHunter(AllocationInstance ai) {
		this.ai = ai;
	}

	public void visit(DiNode<Instruction> n) {
		Instruction inst = n.getData();
		this.checkBadAlloc(inst);
		this.checkInstOvt(inst);
		this.checkCallOvt(inst);
		this.checkUndefAlloc(inst);
	}

	/**
	 * (t, r1) \in def \wedge (t, r2) \in out \wedge r1 \neq r2
	 */
	public void checkBadAlloc(Instruction i) {
		Collection<Register> defSet = i.getDefSet();
		for (Register r1 : defSet) {
			if (!(r1 instanceof BoundReg)) {
				System.err.println("BAD_ALLOC - register not bound: " + r1 + " at " + i);
				this.foundError = true;
			} else {
				BoundReg rb1 = (BoundReg) r1;
				Collection<Register> outSet = i.getOutSet();
				for (Register r2 : outSet) {
					if (!(r2 instanceof BoundReg)) {
						System.err.println("BAD_ALLOC - register not bound: " + r2 + " at " + i);
						this.foundError = true;
					} else {
						BoundReg rb2 = (BoundReg) r2;
						if (rb1.getPseudoReg().equals(rb2.getPseudoReg())) {
							if (!rb1.getMachineReg()
									.equals(rb2.getMachineReg())) {
								System.err.println("BAD_ALLOC: " + rb1
										+ " and " + rb2 + " at " + i);
								this.foundError = true;
							}
						}
					}
				}
			}
		}
	}

	/**
	 * (t2, r) \in out \wedge (t1, r) \in def \wedge t1 \neq t2
	 */
	public void checkInstOvt(Instruction i) {
		Collection<Register> outSet = i.getOutSet();
		for (Register r1 : outSet) {
			if (!(r1 instanceof BoundReg)) {
				System.err.println("INST_OVT - register not bound: " + r1 + " at " + i);
				this.foundError = true;
			} else {
				BoundReg rb1 = (BoundReg) r1;
				Collection<Register> defSet = i.getDefSet();
				for (Register r2 : defSet) {
					if (!(r2 instanceof BoundReg)) {
						System.err.println("INST_OVT - register not bound: " + r2 + " at " + i);
						this.foundError = true;
					} else {
						BoundReg rb2 = (BoundReg) r2;
						if (rb1.getAllocationDestiny().equals(
								rb2.getAllocationDestiny())) {
							if (!rb1.getPseudoReg().equals(rb2.getPseudoReg())) {
								System.err.println("INST_OVT: " + rb1
										+ " and " + rb2 + " at " + i);
								this.foundError = true;
							}
						}
					}
				}
			}
		}
	}

	/**
	 * (t2, r) \in out \wedge (t2, r) \notin def \wedge isCallerSave(r)
	 */
	private void checkCallOvt(Instruction i) {
		if (this.ai.iF.isCall(i)) {
			Collection<Register> outSet = i.getOutSet();
			for (Register r : outSet) {
				if (!(r instanceof BoundReg)) {
					System.err.println("CALL_OVT - register not bound: " + r + " at " + i);
					this.foundError = true;
				} else {
					BoundReg rb = (BoundReg) r;
					if (!i.getDefSet().contains(rb)) {
						if (this.isStoredInCallerSaved(rb)) {
							System.err.println("CALL_OVT: " + r + " at " + i);
							this.foundError = true;
						}
					}
				}
			}
			// Alive across: (t2, r) \in out \wedge (t2, r) \notin def
		}
	}

	private boolean isStoredInCallerSaved(Register r) {
		ArrayList<MachineReg> list = this.ai.rF.getCallerSaveRegisters();
		if (list.contains(r.getAllocationDestiny())) {
			return true;
		}
		return false;
	}

	/**
	 * (t, r) \in in(i0) \wedge t is pseudo register.
	 */
	public void checkUndefAlloc(Instruction i) {

		if(i.getName().equals("0")) {
			Collection<Register> inSet = i.getInSet();			
			for (Register r : inSet) {
				if (!(r instanceof BoundReg)) {
					System.err.println("UNDEF_ALLOC - register not bound: " + r + " at " + i);
					this.foundError = true;
				} else {
					BoundReg rb = (BoundReg) r;
					if( ! rb.getPseudoReg().isPreColoredRegister() ) {
						System.err.println("UNDEF_ALLOC: " + r + " at " + i);
						this.foundError = true;
					}
				}
			}
		}
	}

	public boolean isSemanticallyConsistent() {
		return !foundError;
	}

	public void printAllocation(Instruction inst) {
		System.err.print("\n" + inst.getName() + ": ");
		for (Register reg : inst.getDefSet()) {
			String allocation = (reg.getAllocationDestiny() == null) ? "<SPILL>"
					: reg.getAllocationDestiny().getName();
			System.err.print(reg.getName() + "(" + allocation + ") ");
		}

		System.err.print(" <-- ");
		for (Register reg : inst.getUseSet()) {
			String allocation = (reg.getAllocationDestiny() == null) ? "<SPILL>"
					: reg.getAllocationDestiny().getName();
			System.err.print(reg.getName() + "(" + allocation + ") ");
		}

		System.err.print(" In: ");
		for (Register reg : inst.getInSet()) {
			String allocation = (reg.getAllocationDestiny() == null) ? "<SPILL>"
					: reg.getAllocationDestiny().getName();
			System.err.print(reg.getName() + "(" + allocation + ") ");
		}

		System.err.print(" Out ");
		for (Register reg : inst.getOutSet()) {
			String allocation = (reg.getAllocationDestiny() == null) ? "<SPILL>"
					: reg.getAllocationDestiny().getName();
			System.err.print(reg.getName() + "(" + allocation + ") ");
		}

		System.err.println();
	}

}