package RegAlloc.KrParser;

import java.io.IOException;
import java.util.LinkedList;
import java.util.StringTokenizer;

import Digraph.DiNode;
import Digraph.DiNodeFc;
import Digraph.SimpleDigraph;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.InstructionFactory;
import RegAlloc.MachineReg;
import RegAlloc.PseudoReg;
import RegAlloc.Register;
import RegAlloc.RegisterFactory;
import RegAlloc.RalfTools.ControlFlowShrinker;


/**
 * This class is a parser for the files generated by Krishna's framework. The
 * information read will be stored in an structure of the type
 * <CODE>AllocationInstance</CODE>. This structure contains the control flow
 * graph, and the interference graph generated.
 */
public class InKr {

    public static AllocationInstance read(String fileName) {

        // initialized the input data:
        AllocationInstance allocationInstance = new AllocationInstance();
        allocationInstance.cfg = new SimpleDigraph<Instruction>();
        allocationInstance.cfg.setNodeFactory(new DiNodeFc<Instruction>());
        allocationInstance.rF = new RegisterFactory();
        allocationInstance.iF = new InstructionFactory();
        allocationInstance.lsm = new LinkedList<PseudoReg>();

        try {
            java.io.BufferedReader fileReader = new java.io.BufferedReader(new java.io.FileReader(fileName));

            // creates the pseudo registers:
            skipUntil(fileReader, "set pseudos :=");
            String line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                allocationInstance.rF.createPseudoRegister(line);
                line = fileReader.readLine();
            }


            // read the set of instructions:
            skipUntil(fileReader, "set insts :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                Instruction inst = allocationInstance.iF.createInstruction(line);
                allocationInstance.cfg.newNode(inst);
                line = fileReader.readLine();
            }


            // creates the machine registers:
            skipUntil(fileReader, "set regs :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                allocationInstance.rF.createMachineRegister(line);
                line = fileReader.readLine();
            }


            // read the set of caller saved registers:
            skipUntil(fileReader, "param: callerSave :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                allocationInstance.rF.makeCallerSave(st.nextToken());
                line = fileReader.readLine();
            }


            // read the frequency of instruction use
            skipUntil(fileReader, "param: freq :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst = allocationInstance.iF.createInstruction(st.nextToken());
                int frequency = Integer.parseInt(st.nextToken());
                inst.setFrequency(frequency);
                line = fileReader.readLine();
            }


            // read the liveness data of pseudo-registers. A line is in the
            // format INST REG 1, meaning that register REG is alive in
            // instruction INST.
            skipUntil(fileReader, "param: Live :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst =  allocationInstance.iF.createInstruction(st.nextToken());
                PseudoReg reg = allocationInstance.rF.createPseudoRegister(st.nextToken());
                inst.addIn(reg);
                line = fileReader.readLine();
            }


            // read the liveness data of pre-colored registers:
            skipUntil(fileReader, "param: LiveHardReg :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst =  allocationInstance.iF.createInstruction(st.nextToken());
                MachineReg reg = allocationInstance.rF.createMachineRegister(st.nextToken());
                inst.addIn(reg);
                line = fileReader.readLine();
            }


            // read the list of uses. Each line is in the format INST REG 1,
            // meaning that register REG is used in instruction INST.
            skipUntil(fileReader, "param: req :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst = allocationInstance.iF.createInstruction(st.nextToken());
                Register reg = allocationInstance.rF.createPseudoRegister(st.nextToken());
                inst.addUse(reg);
                reg.addUse(inst);
                line = fileReader.readLine();
            }


            // read the list of uses of pre-colored registers
            skipUntil(fileReader, "param: HRreq :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst = allocationInstance.iF.createInstruction(st.nextToken());
                Register reg = allocationInstance.rF.createMachineRegister(st.nextToken());
                inst.addUse(reg);
                reg.addUse(inst);
                line = fileReader.readLine();
            }


            // read the sequential precedence data:
            skipUntil(fileReader, "param: prevInst :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction i1 = allocationInstance.iF.createInstruction(st.nextToken());
                Instruction i2 = allocationInstance.iF.createInstruction(st.nextToken());
                allocationInstance.cfg.connect(i1, i2);
                line = fileReader.readLine();
            }


            // read the precedence data among branches:
            skipUntil(fileReader, "param: prevInsts :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction i1 = allocationInstance.iF.createInstruction(st.nextToken());
                Instruction i2 = allocationInstance.iF.createInstruction(st.nextToken());
                allocationInstance.cfg.connect(i1, i2);
                line = fileReader.readLine();
            }


            // read the definitions of pseudo registers
            skipUntil(fileReader, "param: def :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst = allocationInstance.iF.createInstruction(st.nextToken());
                Register reg = allocationInstance.rF.createPseudoRegister(st.nextToken());
                inst.addDef(reg);
                reg.addDef(inst);
                line = fileReader.readLine();
            }


            // read the definitions of pre-colored registers
            skipUntil(fileReader, "param: defHardReg :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst = allocationInstance.iF.createInstruction(st.nextToken());
                Register reg = allocationInstance.rF.createMachineRegister(st.nextToken());
                inst.addDef(reg);
                reg.addDef(inst);
                line = fileReader.readLine();
            }


            // read the LMS Pseudos:
            skipUntil(fileReader, "param: lsmPseudos :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer tk = new StringTokenizer(line);
                PseudoReg pseudo = allocationInstance.rF.createPseudoRegister(tk.nextToken());
                allocationInstance.lsm.add(pseudo);
                line = fileReader.readLine();
            }


            // read the known loads. Known loads represent implicit load
            // instructions. Each line is in the format: INST REG 1. It means
            // that the register REG must be loaded in instruction INST.
            skipUntil(fileReader, "param: knownLoads :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer st = new StringTokenizer(line);
                Instruction inst = allocationInstance.iF.createInstruction(st.nextToken());
                Register pseudo = allocationInstance.rF.createPseudoRegister(st.nextToken());
                inst.addDef(pseudo);
                pseudo.addDef(inst);
                line = fileReader.readLine();
            }


            // read the jump instructions:
            skipUntil(fileReader, "param: jumpInst :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer tk = new StringTokenizer(line);
                allocationInstance.iF.makeJumpInst(tk.nextToken());
                line = fileReader.readLine();
            }


            // read the call instructions:
            skipUntil(fileReader, "param: callInst :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer tk = new StringTokenizer(line);
                String instName = tk.nextToken();
                int instNumber = Integer.parseInt(instName);
                if(instNumber % 2 == 0)
                    allocationInstance.iF.makeCallInst(instName);
                line = fileReader.readLine();
            }

            // read the move instructions:
            skipUntil(fileReader, "param: Moves :=");
            line = fileReader.readLine();
            while(line.charAt(0) != ';') {
                StringTokenizer tk = new StringTokenizer(line);
                allocationInstance.iF.makeMoveInst(tk.nextToken());
                line = fileReader.readLine();
            }

            removeOddInstructions(allocationInstance);

        } catch (Exception ioe) {
            ioe.printStackTrace();
        }
        return allocationInstance;
    }


    private static void removeOddInstructions(AllocationInstance ai) {
        Instruction i = ai.iF.createInstruction("0");
        DiNode<Instruction> n = ai.cfg.getNode(i);

        ControlFlowShrinker cfs = new ControlFlowShrinker();
        cfs.setGraph(ai.cfg);
        cfs.traverse(n);
    }


    public static String skipUntil(java.io.BufferedReader fileReader, String text) throws IOException {
        String line = fileReader.readLine();
        while (!line.startsWith(text)) {
            line = fileReader.readLine();
        }
        return line;
    }

}
