package RegAlloc.KrParser;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.StringTokenizer;

import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.PseudoReg;
import RegAlloc.RegisterBinding;
import RegAlloc.RegisterPair;

public class FordReaderImpl implements FordReader {

	private AllocationInstance allocationInstance = null;

	private LinkedList<RegisterBinding> xdf = null;

	private LinkedList<RegisterBinding> psr = null;

	private LinkedList<RegisterBinding> spl = null;

	private LinkedList<RegisterPair> ldp = null;

	private LinkedList<RegisterBinding> sps = null;

	private LinkedList<RegisterPair> stp = null;

	private LinkedList<RegisterPair> psl = null;

	private HashMap<RegisterPair, Boolean> inl = null;

	private HashMap<RegisterPair, Boolean> ins = null;

	private LinkedList<MachineReg> frg = null;

	private LinkedList<RegisterPair> rmv = null;

	public Collection<RegisterBinding> readXDef() {
		return this.xdf;
	}

	public Collection<RegisterBinding> readPsR() {
		return this.psr;
	}

	public Collection<RegisterBinding> readSpLoad() {
		return this.spl;
	}

	public Collection<RegisterPair> readLoadPair() {
		return this.ldp;
	}

	public Collection<RegisterBinding> readSpStore() {
		return this.sps;
	}

	public Collection<RegisterPair> readStorePair() {
		return this.stp;
	}

	public Collection<RegisterPair> readPseudoLocs() {
		return this.psl;
	}

	public HashMap<RegisterPair, Boolean> readInverseLoad() {
		return this.inl;
	}

	public HashMap<RegisterPair, Boolean> readInverseStore() {
		return this.ins;
	}

	public Collection<MachineReg> readFreeRegs() {
		return this.frg;
	}

	public Collection<RegisterPair> readRegMoves() {
		return this.rmv;
	}

	public FordReaderImpl(String fileName, AllocationInstance allocationInstance) {
		this.allocationInstance = allocationInstance;

		this.initializeDataStructures();

		try {
			BufferedReader fileReader = new BufferedReader(new FileReader(
					fileName));

			// read xdef
			skipUntil(fileReader, "xdef:=");
			String line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				RegisterBinding rb = this.readRegisterBinding(line);
				this.checkBinding(rb, true);
				this.xdf.add(rb);
				line = fileReader.readLine();
			}

			// read psr
			skipUntil(fileReader, "PsR:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				RegisterBinding rb = this.readRegisterBinding(line);
				this.checkBinding(rb, false);
				this.psr.add(rb);
				line = fileReader.readLine();
			}

			// read spLoad
			skipUntil(fileReader, "spLoad:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				RegisterBinding rb = this.readRegisterBinding(line);
				this.spl.add(rb);
				line = fileReader.readLine();
			}

			// read load pair
			skipUntil(fileReader, "loadPair:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

			// read sp store
			skipUntil(fileReader, "spStore:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				RegisterBinding rb = this.readRegisterBinding(line);
				this.sps.add(rb);
				line = fileReader.readLine();
			}

			// read store pair
			skipUntil(fileReader, "storePair:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

			skipUntil(fileReader, "f:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

			// read inverse load
			skipUntil(fileReader, "inverseLoad:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

			// read inverse store
			skipUntil(fileReader, "inverseStore:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

			// read free regs
			skipUntil(fileReader, "freeRegs:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

			// read reg moves
			skipUntil(fileReader, "RegMoves:=");
			line = fileReader.readLine();
			while (line.charAt(0) != ';') {
				line = fileReader.readLine();
			}

		} catch (IOException ioe) {
			ioe.printStackTrace();
		}

	}

	public RegisterBinding readRegisterBinding(String line) {
		StringTokenizer st = new StringTokenizer(line);
		String iName = st.nextToken();
		if (!this.allocationInstance.iF.isValidInstruction(iName)) {
			System.err.println("Sintax error: instruction " + iName + " in the ford file is not a valid instruction.");
		}
		Instruction it = allocationInstance.iF.createInstruction(iName);
		String pName = st.nextToken();
		if(!this.allocationInstance.rF.isValidPseudoRegister(pName)) {
			System.err.println("Sintax error: temporary " + pName + " in the ford file is not a valid temporary.");			
		}
		PseudoReg pr = allocationInstance.rF.createPseudoRegister(pName);
		String mName = st.nextToken();
		if(!this.allocationInstance.rF.isValidMachineRegister(mName)) {
			System.err.println("Sintax error: machine register " + mName + " in the ford file is not a valid register.");			
		}
		MachineReg mr = allocationInstance.rF.createMachineRegister(mName);
		return new RegisterBinding(it, pr, mr);
	}

	public void checkBinding(RegisterBinding rb, boolean isDef) {
		Instruction it = rb.getInstruction();
		PseudoReg pr = rb.getPseudoReg();
		if(isDef) {
			if(!it.getDefSet().contains(pr)) {
				System.err.println("Sintax error: the temporary " + pr + " is not in the definition set of " + it);
			}
		} else {
			if(!it.getUseSet().contains(pr)) {
				System.err.println("Sintax error: the temporary " + pr + " is not in the use set of " + it);				
			}
		}		
	}

	public String skipUntil(BufferedReader fileReader, String text)
			throws IOException {
		String line = fileReader.readLine();
		while (!line.startsWith(text)) {
			line = fileReader.readLine();
		}
		return line;
	}

	public void initializeDataStructures() {
		this.xdf = new LinkedList<RegisterBinding>();
		this.psr = new LinkedList<RegisterBinding>();
		this.spl = new LinkedList<RegisterBinding>();
		this.ldp = new LinkedList<RegisterPair>();
		this.sps = new LinkedList<RegisterBinding>();
		this.stp = new LinkedList<RegisterPair>();
		this.psl = new LinkedList<RegisterPair>();
		this.inl = new HashMap<RegisterPair, Boolean>();
		this.ins = new HashMap<RegisterPair, Boolean>();
		this.frg = new LinkedList<MachineReg>();
		this.rmv = new LinkedList<RegisterPair>();
	}

	public String toString() {
		String ans = "";
		ans += this.toStringAux(this.xdf, "xdef:=");
		ans += this.toStringAux(this.psr, "PsR:=");
		ans += this.toStringAux(this.spl, "spLoad:=");
		ans += this.toStringAux(this.sps, "spStore:=");
		return ans;
	}

	private <E> String toStringAux(Collection<E> c, String t) {
		String ans = t + "\n";
		for (E e : c)
			ans = ans + e + "\n";
		return ans;
	}

}