package RegAlloc.KrParser;

import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import utilities.Useful;
import Digraph.DepthFirstSearch;
import Digraph.DiNode;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.MemoryAccessInstruction;
import RegAlloc.PseudoReg;
import RegAlloc.Register;

public class FordPrinter extends DepthFirstSearch<Instruction> {

	private PrintStream out = null;

	private AllocationInstance ai;

	public Set<OutputLine> stores;

	public Set<OutputLine> loads;

	public Set<OutputLine> xdefs;

	public Set<OutputLine> psr;

	public FordPrinter(AllocationInstance ai) {
		this.ai = ai;
		stores = new HashSet<OutputLine>();
		loads = new HashSet<OutputLine>();
		xdefs = new HashSet<OutputLine>();
		psr = new HashSet<OutputLine>();
	}

	public FordPrinter(AllocationInstance ai, String fileName) {
		try {
			this.out = new PrintStream(fileName);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		this.ai = ai;
		stores = new HashSet<OutputLine>();
		loads = new HashSet<OutputLine>();
		xdefs = new HashSet<OutputLine>();
		psr = new HashSet<OutputLine>();
	}

	public void print() {
		Instruction startInst = this.ai.iF.createInstruction("0");
		DiNode<Instruction> startNode = ai.cfg.getNode(startInst);
		this.traverse(startNode);
		this.printOutput();
	}

	public void visit(DiNode<Instruction> n) {
		Instruction inst = n.getData();

		if (inst instanceof MemoryAccessInstruction) {
			MemoryAccessInstruction mai = inst.getMemoryAccessEqv();
			Instruction originInst = mai.getOrigin();
			int index = Integer.parseInt(originInst.getName());
			// if there is use, it is a store
			ArrayList<Register> uses = mai.getUseSet();
			for (Register reg : uses) {
				int regLocIn = reg.getAllocationDestiny().getIntegerName();
				int originRegIn = this.ai.rF.getOriginalRegister(reg)
						.getIntegerName();
				OutputLine ol = new OutputLine(index + " " + originRegIn + " "
						+ regLocIn);
				this.stores.add(ol);
			}
			// if there is def, it is a load
			ArrayList<Register> defs = mai.getDefSet();
			for (Register reg : defs) {
				int regLocIn = reg.getAllocationDestiny().getIntegerName();
				int originRegIn = this.ai.rF.getOriginalRegister(reg)
						.getIntegerName();
				OutputLine ol = new OutputLine(index + " " + originRegIn + " "
						+ regLocIn);
				this.loads.add(ol);
			}
		} else if (this.ai.iF.isCall(inst)) {
			// store the caller save registers before the instruction
			ArrayList<Register> liveList = inst.getInSet();
			for (Register reg : liveList) {
				if (this.mustBeSaved(reg)) {
					Register regLoc = reg.getAllocationDestiny();
					Collection<DiNode<Instruction>> preds = n.preds();
					for (DiNode<Instruction> pred : preds) {
						Instruction storeLoc = pred.getData().getOrigin();
						if (!this.ai.iF.isCall(pred.getData())) {
							OutputLine ol = new OutputLine(storeLoc.getName()
									+ " " + reg.getIntegerName() + " "
									+ regLoc.getIntegerName());
							this.stores.add(ol);
						}
					}
				}
			}
			// load the caller save registers after the instruction
			liveList = inst.getOutSet();
			for (Register reg : liveList) {
				if (this.mustBeSaved(reg)) {
					Register regLoc = reg.getAllocationDestiny();
					Collection<DiNode<Instruction>> succs = n.succs();
					for (DiNode<Instruction> succ : succs) {
						Instruction loadLoc = succ.getData().getOrigin();
						if (!this.ai.iF.isCall(succ.getData())) {
							OutputLine ol = new OutputLine(loadLoc.getName()
									+ " " + reg.getIntegerName() + " "
									+ regLoc.getIntegerName());
							this.loads.add(ol);
						}
					}
				}
			}
		} else {
			// print definitions:
			for (Register def : inst.getDefSet()) {
				if( ! def.isPreColoredRegister() ) {
					String defLoc = inst.getName();
					int tempName = def.getIntegerName();
					int machineRegName = def.getAllocationDestiny()
							.getIntegerName();
					OutputLine ol = new OutputLine(defLoc + " " + tempName + " "
							+ machineRegName);
					this.xdefs.add(ol);					
				}
			}
			// print uses:
			for (Register use : inst.getUseSet()) {
				if( ! use.isPreColoredRegister() ) {
					String useLoc = inst.getName();
					int tempName = use.getIntegerName();
					int machineRegName = use.getAllocationDestiny()
							.getIntegerName();
					OutputLine ol = new OutputLine(useLoc + " " + tempName + " "
							+ machineRegName);
					this.psr.add(ol);					
				}
			}
		}

	}

	/**
	 * This method determines if the pseudo register must be saved before call
	 * instructions. A Pseudo must be saved if it is stored in a caller saved
	 * register. Notice that this rule does not apply to machine registers.
	 */
	private boolean mustBeSaved(Register reg) {
		if (reg.isPreColoredRegister())
			return false;
		else {
			ArrayList<MachineReg> callerSaveRegs = this.ai.rF
					.getCallerSaveRegisters();
			if (callerSaveRegs.contains(reg.getAllocationDestiny()))
				return true;
			else
				return false;
		}
	}

	private void printOrderedList(Set<OutputLine> list) {
		List<OutputLine> sorted = Useful.sort(list);
		HashSet<String> alreadyPrinted = new HashSet<String>();
		for (int i = 0; i < sorted.size(); i++) {
			OutputLine ol = sorted.get(i);
			String output = ol.inst + " " + ol.pseudo + " " + ol.reg;
			if (!alreadyPrinted.contains(output)) {
				alreadyPrinted.add(output);
				this.out.println(output);
			}
		}
	}

    public void printLSM() {
        ListIterator<PseudoReg> iter = this.ai.lsm.listIterator();
        while(iter.hasNext()) {
            PseudoReg reg = iter.next();
            this.out.println(reg.getIntegerName() + " " + reg.getIntegerName());
        }
    }

	private void printOutput() {
		this.out.println("xdef:=");
		printOrderedList(this.xdefs);
		this.out.println(";");

		this.out.println("PsR:=");
		printOrderedList(this.psr);
		this.out.println(";");

		this.out.println("spLoad:=");
		printOrderedList(this.loads);
		this.out.println(";");

		this.out.println("loadPair:=");
		this.out.println(";");

		this.out.println("spStore:=");
		printOrderedList(this.stores);
		this.out.println(";");

		this.out.println("storePair:=");
		this.out.println(";");

		this.out.println("f:=");
		this.printLSM();
		this.out.println(";");

		this.out.println("inverseLoad:=");
		this.out.println(";");

		this.out.println("inverseStore:=");
		this.out.println(";");

		this.out.println("freeRegs:=");
		this.out.println(";");

		this.out.println("RegMoves:=");
		this.out.println(";");
	}

	public void setOut(PrintStream out) {
		this.out = out;
	}

}
