package RegAlloc;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;

/**
 * The instruction factory is responsible for the creation of objects
 * representing instructions. The advantage of using an object factory is the
 * ability of keeping all the creation points in a single piece of code. This
 * factory keeps all the instructions created in a list. It also keeps track of
 * which instructions are jumps and calls.
 */
public class InstructionFactory {

	private HashMap<String, Instruction> instList = new HashMap<String, Instruction>();

	private HashSet<MemoryAccessInstruction> memoryAccessTable = new HashSet<MemoryAccessInstruction>();

	private LinkedList<Instruction> callInst = new LinkedList<Instruction>();

	private LinkedList<Instruction> jumpInst = new LinkedList<Instruction>();

	private LinkedList<Instruction> moveInst = new LinkedList<Instruction>();

	/**
	 * This method informs if the given string represents the identifier of a
	 * valid instruction.
	 * @param iName the string that may represent an instruction
	 * @return true if the parameter represents an instruction stored in this
	 * data structure, and false otherwise.
	 */
	public boolean isValidInstruction(String iName) {
		return this.instList.containsKey(iName);
	}

	/**
	 * The so called memory access instructions are special instructions created
	 * to hold a load or store generated due to spilling of registers. The
	 * method <CODE>getInstructions</CODE> does not return the memory access
	 * instructions.
	 */
	public MemoryAccessInstruction createMemoryAccessInstruction(String id) {
		Instruction origin = this.instList.get(id);
		MemoryAccessInstruction i = new MemoryAccessInstruction(id, origin);
		this.memoryAccessTable.add(i);
		return i;
	}

	public Collection<MemoryAccessInstruction> getMemoryAccessInstructions() {
		return this.memoryAccessTable;
	}

	public Instruction createInstruction(String id) {
		if (this.instList.containsKey(id))
			return this.instList.get(id);
		Instruction i = new Instruction(id);
		this.instList.put(id, i);
		return i;
	}

	public Collection<Instruction> getInstructions() {
		return this.instList.values();
	}

	/**
	 * @return true if the instruction is a call.
	 */
	public boolean isCall(Instruction inst) {
		return callInst.contains(inst);
	}

	/**
	 * @return true if the instruction is a jump.
	 */
	public boolean isJump(Instruction inst) {
		return jumpInst.contains(inst);
	}

	/**
	 * @return true if the instruction is a move related instruction.
	 */
	public boolean isMove(Instruction inst) {
		return this.moveInst.contains(inst);
	}

	/**
	 * @return a list containing all the move instructions.
	 */
	public LinkedList<Instruction> getMoveInstructions() {
		LinkedList<Instruction> list = new LinkedList<Instruction>();
		for (Instruction inst : this.moveInst) {
			list.add(inst);
		}
		return list;
	}

	/**
	 * This method marks the given instruction as a call. It is important to
	 * know which instructions are function calls in order to save the caller-
	 * save registers.
	 * 
	 * @param id
	 *            the identifier of an instruction. If this string is the
	 *            identifier of an existing instruction, this instruction will
	 *            be made a call instruction.
	 */
	public void makeCallInst(String id) {
		if (!this.instList.containsKey(id))
			this.createInstruction(id);
		this.callInst.add(this.instList.get(id));
	}

	/**
	 * This method marks the given instruction as a jump.
	 * 
	 * @param id
	 *            the identifier of an instruction. If this string is the
	 *            identifier of an existing instruction, this instruction will
	 *            be marked as a jump instruction.
	 */
	public void makeJumpInst(String id) {
		if (!this.instList.containsKey(id))
			this.createInstruction(id);
		this.jumpInst.add(this.instList.get(id));
	}

	/**
	 * This method marks the given instruction as a move.
	 * 
	 * @param id
	 *            the identifier of an instruction. If this string is the
	 *            identifier of an existing instruction, this instruction will
	 *            be marked as a move instruction.
	 */
	public void makeMoveInst(String id) {
		if (!this.instList.containsKey(id))
			this.createInstruction(id);
		this.moveInst.add(this.instList.get(id));
	}

	public String toString() {
		String m = "Instruction set: ";
		Collection<Instruction> isc = this.instList.values();
		for (Instruction mr : isc)
			m += mr.getName() + " ";

		String move = "Move Instructions: ";
		for (Instruction inst : this.moveInst)
			move += inst.getName() + " ";

		String j = "Jump Instructions: ";
		for (Instruction inst : this.jumpInst)
			j += inst.getName() + " ";

		String c = "Call Instructios: ";
		for (Instruction inst : this.callInst)
			c += inst.getName() + " ";

		return m + '\n' + move + '\n' + j + '\n' + c;
	}

}
