package RegAlloc;

import java.util.ArrayList;

/**
 * This class represents an instruction in the control flow graph. Every
 * instruction contains a list of uses, definitions, the in set, and the out
 * set. Instructions also have an identifier, that is, a name that can be
 * used to discriminate them.
 */
public class Instruction {

    private int frequency = 0;

    private String id = null;

    private ArrayList<Register> use = null;
    private ArrayList<Register> def = null;
    private ArrayList<Register> in  = null;
    private ArrayList<Register> out = null;

    public Instruction(String id) {
        this.id = id;
        this.use = new ArrayList<Register>();
        this.def = new ArrayList<Register>();
        this.in  = new ArrayList<Register>();
        this.out = new ArrayList<Register>();
        this.frequency = 1;
    }

    public void setFrequency(int frequency) {
        this.frequency = frequency;
    }

    public int getFrequency() {
        return this.frequency;
    }

    public void addIn(Register reg) {
        this.in.add(reg);
    }

    public void addOut(Register reg) {
        this.out.add(reg);
    }

    public boolean removeUse(Register reg) {
        return this.use.remove(reg);
    }

    public void addUse(Register reg) {
        this.use.add(reg);
    }

    public boolean removeDef(Register reg) {
        return this.def.remove(reg);
    }

    public void addDef(Register reg) {
        this.def.add(reg);
    }

    public ArrayList<Register> getDefSet() {
        return this.def;
    }

    public ArrayList<Register> getUseSet() {
        return this.use;
    }

    public ArrayList<Register> getInSet() {
        return this.in;
    }

    public ArrayList<Register> getOutSet() {
        return this.out;
    }

    public boolean cleanInSet() {
        boolean wasEmpty = this.in.isEmpty();
        this.in = new ArrayList<Register>();
        return wasEmpty;
    }

    public boolean cleanOutSet() {
        boolean wasEmpty = this.out.isEmpty();
        this.out = new ArrayList<Register>();
        return wasEmpty;
    }

    /**
     * This method returns only the name of the instruction. In order to print
     * the full content of the instruction, the method <CODE>toString()</CODE>
     * must be invoked.
     * @return the name of the instruction.
     */
    public String getName() {
        return this.id;
    }

    public String toString() {
        final int DISP = 16;
        String d = "", u = "", n = this.id;

        for(Register r : this.def)
            d += " " + r.getName();
        for(int j = d.length(); j < DISP/2; j++)
            d += " ";

        for(Register r : this.use)
            u += " " + r.getName();
        for(int j = u.length(); j < DISP; j++)
            u += " ";

        for(int j = n.length(); j < DISP/4; j++)
            n += " ";

        return n + " |" + d + " |" + u;
    }

    public String getLabel(boolean showAllocation) {
        ArrayList<Register> def = this.getDefSet();
        String defN = "";
        for(Register r : def) {
            defN += " " + r.getName();
            if(!r.isPreColoredRegister() && showAllocation)
                defN += "(" + r.getAllocationDestiny().getName() + ")";
        }

        ArrayList<Register> use = this.getUseSet();
        String useN = "";
        for(Register r : use) {
            useN += " " + r.getName();
            if(!r.isPreColoredRegister() && showAllocation)
                useN += "(" + r.getAllocationDestiny().getName() + ")";
        }

        ArrayList<Register> in = this.getInSet();
        String inN = "";
        for(Register r : in)
            inN += " " + r.getName();
        return "\"" + this.getName() + " : " + defN + " <= " + useN + "\"";
    }


    /**
     * This method is used to avoid the use of type casts. In this way, an
     * object whose dynamic type is a Memory Access, but whose static type is a
     * simple instruction can be converted to memory access during execution
     * time.
     */
    public MemoryAccessInstruction getMemoryAccessEqv() {
        return null;
    }

    /**
     * During the code generation process, some 'fake' instructions are
     * produced to hold load's and store's. When this method is invoked on a
     * such 'fake' instruction, it returns the original instruction, that is,
     * the instruction where the spill happened. When it is invoked on a
     * actual instruction, it returns the instruction itself;
     */
    public Instruction getOrigin() {
        return this;
    }


    public String extToString() {
        final int DISP = 16;
        String d = "", u = "", i = "", o = "", n = this.id;

        for(Register r : this.def)
            d += " " + r.getName();
        for(int j = d.length(); j < DISP/2; j++)
            d += " ";

        for(Register r : this.use)
            u += " " + r.getName();
        for(int j = u.length(); j < DISP; j++)
            u += " ";

        for(Register r : this.in)
            i += " " + r.getName();
        for(int j = i.length(); j < DISP*2; j++)
            i += " ";

        for(Register r : this.out)
            o += " " + r.getName();

        for(int j = n.length(); j < DISP/4; j++)
            n += " ";

        return n + " |" + d + " |" + u + " |" + i + " |" + o;
    }

}
