package RegAlloc.IRC;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Stack;

import utilities.ListOp;
import Digraph.DiNode;
import Graph.Node;
import Graph.SimpleGraph;
import Graph.SimpleNodeFc;
import RegAlloc.AllocationData;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.MemoryAccessInstruction;
import RegAlloc.MemoryAccessRegister;
import RegAlloc.PseudoReg;
import RegAlloc.RegAllocator;
import RegAlloc.Register;
import RegAlloc.KrParser.InKr;
import RegAlloc.KrParser.OutKr;
import RegAlloc.RalfTools.DFA;
import RegAlloc.RalfTools.GraphvizGen;

/**
 * This class contains an implementation of the algorithm Iterated Register
 * Coalescing, of George and Appel. A detailed description of the algorithm
 * can be found in the paper of same name, in ACM Transactions on Programming
 * Languages and Systems, 300 - 324, Volume 18, Issue 3, Year 1996.
 */
public class IRC_Alloc implements RegAllocator {

    // contains the control flow graph, interference graph, register and
    // instruction factories
    private AllocationInstance ai = null;

    // This graph is a clone of the interference graph. It is used to keep
    // track of the interference relations when checking coalescing.
    private SimpleGraph<Register> auxiliaryGraph = null;

    /*
      The following lists and sets are always mutually disjoint and every
      node is always in exactly one of the sets or lists.
     */

    // temporary registers, not precolored and not yet processed
    private Collection<Register> initial = new HashSet<Register>();

    // list of low-degree non-move-related nodes.
    private Collection<Register> simplifyWorkList = new HashSet<Register>();

    // low-degree move-related nodes.
    private Collection<Register> freezeWorkList = new HashSet<Register>();

    // high-degree nodes.
    private Collection<Register> spillWorkList = new HashSet<Register>();

    // nodes marked for spilling during the whole algorithm; initially empty.
    private Collection<Register> spilledNodes = new HashSet<Register>();

    // nodes marked for spilling during this round; initially empty.
    private Collection<Register> lastRoundSpill = new HashSet<Register>();

    // registers that have been coalesced; when u := v is coalesced, v is
    // added to this set and u put back on some work list.
    private Collection<Register> coalescedNodes = new HashSet<Register>();

    // nodes successfully colored.
    private Collection<Register> coloredNodes = new HashSet<Register>();

    // stack containing temporaries removed from the graph.
    private Stack<Node<Register>> selectStack = new Stack<Node<Register>>();

    /*
      MOVE SETS: there are five sets of move instructions, and every move is
      in exactly one of these sets (after build through the end of Main).
     */

    // moves that have been coalesced.
    private Collection<Instruction> coalescedMoves = new HashSet<Instruction>();

    // moves whose source and target interfere.
    private Collection<Instruction> constrainedMoves = new HashSet<Instruction>();

    // moves that will no longer be considered for coalescing.
    private Collection<Instruction> frozenMoves = new HashSet<Instruction>();

    // moves enabled for possible coalescing.
    private Collection<Instruction> workListMoves = new HashSet<Instruction>();

    // moves not yet ready for coalescing.
    private Collection<Instruction> activeMoves = new HashSet<Instruction>();

    /* Other data structures */

    // when a move (u, v) has been coalesced, and v put in coalescedNodes,
    // then alias(v) = u;
    private HashMap<Register, Register> alias = new HashMap<Register, Register>();

    private HashMap<Register, Collection<Instruction>> moveList = new HashMap<Register, Collection<Instruction>>();

    /**
     * This method tries to find a coloring to the graph. It loops, via tail
     * recursion, until no spills are generated.
     */
    public void performRegAllocation() {
        this.livenessAnalysis();
        this.build();
        this.makeWorkList();
        do {
            if(this.simplifyWorkList.size() != 0) {
                this.simplify();
            } else if(this.workListMoves.size() != 0) {
                this.coalesce();
            } else if(this.freezeWorkList.size() != 0) {
                this.freeze();
            } else if(this.spillWorkList.size() != 0) {
                this.selectSpill();
            }
        } while ( this.simplifyWorkList.size() != 0 ||
                  this.workListMoves.size() != 0    ||
                  this.freezeWorkList.size() != 0   ||
                  this.spillWorkList.size() != 0
                );
        this.assignColors();
        if(this.lastRoundSpill.size() > 0) {
            this.rewriteProgram();
            performRegAllocation();
        }
    }

    public void livenessAnalysis() {
        DFA.cleanControlFlowGraph(this.ai, "0");
        DFA.performLivenessAnalysis(this.ai, "0");
        DFA.computeLivenessSize(this.ai, "0");
    }

    /**
     * Procedure build constructs the interference graph using the results of
     * static liveness analysis, and also initializes the workListMoves to
     * contain all the moves in the program.
     */
    private void build() {
        this.ai.ig = new SimpleGraph<Register>();
        this.ai.ig.setNodeFactory(new SimpleNodeFc<Register>());

        // build the auxiliary graph
        this.auxiliaryGraph = new SimpleGraph<Register>();
        this.auxiliaryGraph.setNodeFactory(new SimpleNodeFc<Register>());

        // Add all the pseudo registers to the graph.
        ArrayList<PseudoReg> pseudoSet = this.ai.rF.getPseudoRegisters();
        for(PseudoReg pseudo : pseudoSet) {
            if(!this.spilledNodes.contains(pseudo)) {
                this.ai.ig.newNode(pseudo);
                pseudo.resetColor();
            }
        }

        // builds the move lists
        this.workListMoves = new HashSet<Instruction>();
        this.coalescedMoves = new HashSet<Instruction>();
        this.constrainedMoves = new HashSet<Instruction>();
        this.frozenMoves = new HashSet<Instruction>();
        this.activeMoves = new HashSet<Instruction>();
        for(Register reg : this.ai.rF.getRegisters()) {
            this.moveList.put(reg, new LinkedList<Instruction>());
        }

        // set the interferences between simultaneously alive registers
        Collection<DiNode<Instruction>> insts = this.ai.cfg.getNodes();
        for(DiNode<Instruction> nInst : insts) {
            Instruction inst = nInst.getData();
            ArrayList<Register> ll = inst.getInSet();
            for(int j = 0; j < ll.size(); j++) {
                Register r1 = ll.get(j);
                for(int k = j + 1; k < ll.size(); k++) {
                    Register r2 = ll.get(k);
                    this.ai.ig.connect(r1, r2);
                }
            }
            if(this.ai.iF.isMove(inst)) {
                this.workListMoves.add(inst);
                Register x = ListOp.first(inst.getDefSet());
                this.moveList.get(x).add(inst);
                Register y = ListOp.first(inst.getUseSet());
                this.moveList.get(y).add(inst);
            }
        }

        // make the machine registers pairwise interfering
        ArrayList<MachineReg> regSet = this.ai.rF.getMachineRegisters();
        for(int i = 0; i < regSet.size() - 1; i++)
            for(int j = i+1; j < regSet.size(); j++)
                this.ai.ig.connect(regSet.get(i), regSet.get(j));

    }

    private void makeWorkList() {
        // fill the initial collection of registers to be processed
        this.initial = this.nodes2Registers(this.ai.ig.getNodes());
        Iterator<Register> i = this.initial.iterator();
        while(i.hasNext()) {
            Register reg = i.next();
            Node<Register> n = this.ai.ig.getNode(reg);
            if (n.degree() >= this.ai.rF.getNumMachineRegisters()) {
                this.spillWorkList.add(reg);
            } else if(this.moveRelated(reg)) {
                this.freezeWorkList.add(reg);
            } else {
                this.simplifyWorkList.add(reg);
            }
            i.remove();
        }
    }

    private boolean moveRelated(Register reg) {
        Collection<Instruction> defInsts = reg.getDefInstructions();
        for(Instruction inst : defInsts)
            if(this.ai.iF.isMove(inst))
                return true;
        Collection<Instruction> useInsts = reg.getUseInstructions();
        for(Instruction inst : useInsts)
            if(this.ai.iF.isMove(inst))
                return true;
        return false;
    }

    private Collection<Register> adjacent(Node<Register> n) {
        Collection<Register> adjReg = this.nodes2Registers(n.neighbors());
        Collection<Register> stkReg = this.nodes2Registers(this.selectStack);
        return ListOp.df(adjReg, ListOp.un(stkReg, this.coalescedNodes));
    }

    /**
     * For each instruction move-related to v, replace v by u
     */
    private void updateMoveList(Register v, Register u) {
        Collection<Instruction> moveListU = this.moveList.get(u);
        Collection<Instruction> moveListV = this.moveList.get(v);
        for(Instruction inst : moveListV)
            moveListU.add(inst);
    }

    private Collection<Instruction> nodeMoves(Register n) {
        return ListOp.it(this.moveList.get(n), ListOp.un(this.activeMoves, this.workListMoves));
    }

    private void simplify() {
        Register n = ListOp.first(this.simplifyWorkList);
        this.simplifyWorkList.remove(n);
        Node<Register> node = this.ai.ig.remove(n);
        this.selectStack.push(node);
        for(Register m : this.adjacent(node))
            this.decrementDegree(this.ai.ig.getNode(m));
    }

    private void decrementDegree(Node<Register> m) {
        Register reg = m.getData();
        int d = m.degree();
        if(d < this.ai.rF.getNumMachineRegisters()) {
            Collection<Register> adjs = this.adjacent(m);
            adjs.add(reg);
            this.enableMoves(adjs);
            this.spillWorkList.remove(reg);
            if(this.moveRelated(reg))
                this.freezeWorkList.add(reg);
            else
                this.simplifyWorkList.add(reg);
        }
    }

    private void enableMoves(Register reg) {
        for(Instruction m : this.nodeMoves(reg))
            if(this.activeMoves.contains(m)) {
                this.activeMoves.remove(m);
                this.workListMoves.add(m);
            }
    }

    private void enableMoves(Collection<Register> regs) {
        for(Register n : regs)
            for(Instruction m : this.nodeMoves(n))
                if(this.activeMoves.contains(m)) {
                    this.activeMoves.remove(m);
                    this.workListMoves.add(m);
                }
    }

    private void addWorkList(Register u) {
        Node<Register> n = this.ai.ig.getNode(u);
        if(!(u instanceof MachineReg) && !(this.moveRelated(u)) && n.degree() < this.ai.rF.getNumMachineRegisters()) {
            this.freezeWorkList.remove(u);
            this.simplifyWorkList.add(u);
        }
    }

    public boolean OK(Register n1, Register n2) {
        Node<Register> t = this.ai.ig.getNode(n1);
        Node<Register> r = this.ai.ig.getNode(n2);
        boolean b1 = t.degree() < this.ai.rF.getNumMachineRegisters();
        boolean b2 = t instanceof MachineReg;
        boolean b3 = t.adj(r);
        return  b1 || b2 || b3;
    }

    /**
     * Let nodes be a set of nodes. This method returns true if the number of
     * elements in nodes with degree equal or greater than K is smaller than K.
     */
    public boolean conservative(Collection<Node<Register>> nodes) {
        int k = 0;
        for(Node<Register> n : nodes)
            if(n.degree() >= this.ai.rF.getNumMachineRegisters())
                k = k + 1;
        return (k < this.ai.rF.getNumMachineRegisters());
    }

    private void coalesce() {
        Instruction m = ListOp.first(this.workListMoves);
        Register x = this.getAlias(ListOp.first(m.getDefSet()));
        Register y = this.getAlias(ListOp.first(m.getUseSet()));
        Node<Register> u, v;
        if(y instanceof MachineReg) {
            u = this.ai.ig.getNode(y);
            v = this.ai.ig.getNode(x);
        } else {
            u = this.ai.ig.getNode(x);
            v = this.ai.ig.getNode(y);
        }
        this.workListMoves.remove(m);
        if(u.equals(v)) {
            this.coalescedMoves.add(m);
            this.addWorkList(u.getData());
        } else if((v.getData() instanceof MachineReg) || u.adj(v)) {
            this.constrainedMoves.add(m);
            this.addWorkList(u.getData());
            this.addWorkList(v.getData());
        } else if( (u.getData() instanceof MachineReg && verifyOK(this.adjacent(v), u)) || (!(u.getData() instanceof MachineReg) && this.conservative(ListOp.un(u.neighbors(), v.neighbors()))) ) {
            this.coalescedMoves.add(m);
            this.combine(u, v);
            this.addWorkList(u.getData());
        } else
            this.activeMoves.add(m);
    }

    private boolean verifyOK(Collection<Register> adjacent_v, Node<Register> u) {
        Register uReg = u.getData();
        for(Register t : adjacent_v)
            if(!this.OK(t, uReg))
                return false;
        return true;
    }

    private void combine(Node<Register> u, Node<Register> v) {
        Register uReg = u.getData();
        Register vReg = v.getData();
System.err.println("Coalescing " + uReg.getName() + " and " + vReg.getName());
        if(this.freezeWorkList.contains(vReg))
            this.freezeWorkList.remove(vReg);
        else
            this.spillWorkList.remove(vReg);
        this.coalescedNodes.add(vReg);
        this.alias.put(vReg, uReg);
        this.updateMoveList(vReg, uReg);
        this.enableMoves(vReg);
        this.ai.ig.remove(vReg);
        for(Register t : this.adjacent(v)) {
            this.ai.ig.connect(t, uReg);
            this.decrementDegree(this.ai.ig.getNode(t));
        }
        if(u.degree() >= this.ai.rF.getNumMachineRegisters() && this.freezeWorkList.contains(u)) {
            this.freezeWorkList.remove(uReg);
            this.spillWorkList.add(uReg);
        }
    }

    private Register getAlias(Register n) {
        if(this.coalescedNodes.contains(n))
            return this.getAlias(alias.get(n));
        else
            return n;
    }

    private void freeze() {
        for(Register u : this.freezeWorkList) {
            this.freezeWorkList.remove(u);
            this.simplifyWorkList.add(u);
            this.freezeMoves(u);
            break;
        }
    }

    private void freezeMoves(Register u) {
        for(Instruction m : this.nodeMoves(u)) {
            Register x = ListOp.first(m.getDefSet());
            Register y = ListOp.first(m.getUseSet());
            Register v;
            if(this.getAlias(y).equals(this.getAlias(u)))
                v = this.getAlias(x);
            else
                v = this.getAlias(y);
            this.activeMoves.remove(m);
            this.frozenMoves.add(m);
            if(this.freezeWorkList.contains(v) && this.nodeMoves(v).isEmpty()) {
                this.freezeWorkList.remove(v);
                this.simplifyWorkList.add(v);
            }
        }
    }

    private void selectSpill() {
        Register nextSpill =  null;
        for(Register m : this.spillWorkList) {
            if(m instanceof MemoryAccessRegister || m instanceof MachineReg) {
                continue;
            } else {
                if(nextSpill == null) {
                    nextSpill = m;
                } else {
                    if(m.spillingFactor() > nextSpill.spillingFactor()) {
                        nextSpill = m;
                    }
                }
            }
        }
        if(nextSpill == null) {
            System.err.println("Error: it is not possible to choose the next node to be spilled.");
            System.exit(1);
        } else {
            this.spillWorkList.remove(nextSpill);
            this.simplifyWorkList.add(nextSpill);
            this.freezeMoves(nextSpill);
        }
    }


    private void assignColors() {
        int maxColor = 0;
        int[] availableColors = new int[this.ai.rF.getNumMachineRegisters()];
        while(!this.selectStack.empty()) {
            Node<Register> n = this.selectStack.pop();
            Register reg = n.getData();
            int color = this.color(n, availableColors);
            if(color == -1) {
System.err.println("Spilled node: " + reg.getName());
                this.lastRoundSpill.add(reg);
                this.spilledNodes.add(reg);
            }
            else {
                this.coloredNodes.add(reg);
                if(color > maxColor)
                    maxColor = color;
            }
        }       
        for(Register n : this.coalescedNodes) {
            n.setColor(this.getAlias(n).getColor());
            this.coloredNodes.add(n);
        }
    }


    private int color(Node<Register> n, int[] availableColors) {
        for(int i = 0; i < availableColors.length; i++)
            availableColors[i] = -1;

        // find an available color.
        for(Node<Register> succ : n.neighbors()) {
            Register sKey = this.getAlias(succ.getData());
            if(sKey.getColor() >= 0) {
                availableColors[sKey.getColor()] = 1;
            }
        }

        for(int i = 0; i < availableColors.length; i++)
            if(availableColors[i] == -1) {
                n.getData().setColor(i);
                break;
            }
        return n.getData().getColor();
    }


    private void rewriteProgram() {
        Collection<Register> newTemps = this.updateControlFlowGraph();
        this.lastRoundSpill = new HashSet<Register>();
        this.initial = ListOp.un(this.coloredNodes, ListOp.un(this.coalescedNodes, newTemps));
        this.coloredNodes = new HashSet<Register>();
        this.coalescedNodes = new HashSet<Register>();
    }


    private Collection<Register> updateControlFlowGraph() {
        Collection<Register> newTemps = new LinkedList<Register>();
        for(Register reg : this.lastRoundSpill) {
            // place stores:
            Collection<Instruction> defInsts = reg.getDefInstructions();
            for(Instruction inst : defInsts) {
                // remove reg from inst
                inst.removeDef(reg);
                // creates a new instruction, where the new register will be inserted.
                MemoryAccessInstruction inst_store = this.ai.iF.createMemoryAccessInstruction(inst.getName());
                // create reg_store as a surrogate to the evicted register.
                MemoryAccessRegister reg_store = this.ai.rF.createMemoryAccessRegister(reg);
                newTemps.add(reg_store);
                inst.addDef(reg_store);
                reg_store.addDef(inst);
                inst_store.addUse(reg_store);
                reg_store.addUse(inst_store);
                // place the new instruction in the control flow graph:
                DiNode<Instruction> node = ai.cfg.getNode(inst);
                Collection<DiNode<Instruction>> succs = node.succs();
                Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
                for(DiNode<Instruction> succ : succs) {
                    toBeRemoved.add(succ);
                    succ.removePred(node);
                    ai.cfg.connect(inst_store, succ.getData());
                }
                for(DiNode<Instruction> succ : toBeRemoved) // avoid concurrent ex
                    node.removeSucc(succ);
                ai.cfg.connect(inst, inst_store);
            }

            // place loads:
            Collection<Instruction> useInsts = reg.getUseInstructions();
            for(Instruction inst : useInsts) {
                // remove reg from inst
                inst.removeUse(reg);
                // creates a new instruction, where the new register will be inserted.
                MemoryAccessInstruction inst_load = this.ai.iF.createMemoryAccessInstruction(inst.getName());
                // create reg_load as a surrogate to the loaded register.
                MemoryAccessRegister reg_load = this.ai.rF.createMemoryAccessRegister(reg);
                newTemps.add(reg_load);
                inst.addUse(reg_load);
                reg_load.addUse(inst);
                inst_load.addDef(reg_load);
                reg_load.addDef(inst_load);
                // place the new instruction in the control flow graph:
                DiNode<Instruction> node = ai.cfg.getNode(inst);
                Collection<DiNode<Instruction>> preds = node.preds();
                Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
                for(DiNode<Instruction> pred : preds) {
                    toBeRemoved.add(pred);
                    pred.removeSucc(node);
                    ai.cfg.connect(pred.getData(), inst_load);
                }
                for(DiNode<Instruction> pred : toBeRemoved) // avoid concurrent ex
                    node.removePred(pred);
                ai.cfg.connect(inst_load, inst);
            }
        }
        return newTemps;
    }

    public Collection<Register> nodes2Registers(Collection<Node<Register>> c) {
        LinkedList<Register> list = new LinkedList<Register>();
        for(Node<Register> n : c)
            list.add(n.getData());
        return list;
    }

    /**
     * This method yields the collection of spilled registers. It must be
     * called after the method <CODE>performRegAllocation()</CODE> is called.
     * @return an collection of nodes of the allocation graph.
     */
    public Collection<Register> getSpilledRegisters() {
        return this.spilledNodes;
    }

    /**
     * This method gives back the list of nodes that have been allocated to
     * any register.
     * @return an collection of nodes of the allocation graph.
     */
    public Collection<Register> getAllocatedRegisters() {
        return this.coloredNodes;
    }

    public void setAllocationParameters(AllocationInstance ai) {
        this.ai = ai;
    }


    /**
     * This method must be called after the register allocation has been
     * performed. It will assign for each pseudo register an actual location.
     */
    public void setFinalLocation() {
        // get the machine registers and color them
        ArrayList<MachineReg> mRegs = this.ai.rF.getMachineRegisters();
        ArrayList<MachineReg> aux = new ArrayList<MachineReg>();
        for(int i = 0; i < mRegs.size(); i++)
            aux.add(mRegs.get(i));

        for(int i = 0; i < mRegs.size(); i++) {
            MachineReg mAux = mRegs.get(i);
            aux.set(mAux.getColor(), mAux);
        }
        // allocate the pseudo registers
        this.allocatePseudos(aux);
    }


    public AllocationData getAllocationData() {
        return new AllocationData();
    }



    /**
     * Each color is associated to a machine register. This method assigns to
     * each pseudo the machine register that is associated to the node's color.
     */
    private void allocatePseudos(ArrayList<MachineReg> regSet) {
        ArrayList<PseudoReg> regs = ai.rF.getPseudoRegisters();
        for(Register reg : regs) {
            if(reg.getColor() < 0) {
                System.err.println("Spilled register: " + reg);
            } else if(reg.getColor() < regSet.size()) {
                reg.setAllocationDestiny(regSet.get(reg.getColor()));
            }
        }
    }


    public void printRegisterLists() {
        utilities.Useful.printCollection("Simplify work list", simplifyWorkList);
        utilities.Useful.printCollection("Coalesce work list", workListMoves);
        utilities.Useful.printCollection("Freeze work list", freezeWorkList);
        utilities.Useful.printCollection("Spill work list", spillWorkList);
    }

    public void dumpLists() {
        utilities.Useful.printCollection("Initial: ",            this.initial);
        utilities.Useful.printCollection("simplify work list: ", this.simplifyWorkList);
        utilities.Useful.printCollection("freeze work list: ",   this.freezeWorkList);
        utilities.Useful.printCollection("spill work list: ",    this.spillWorkList);
        utilities.Useful.printCollection("spilled nodes: ",      this.spilledNodes);
        utilities.Useful.printCollection("last round spill: ",   this.lastRoundSpill);
        utilities.Useful.printCollection("coalesced nodes: ",    this.coalescedNodes);
        utilities.Useful.printCollection("colored nodes: ",      this.coloredNodes);
        utilities.Useful.printCollection("select stack: ",       this.selectStack);
        utilities.Useful.printCollection("coalesced moves: ",    this.coalescedMoves);
        utilities.Useful.printCollection("constrained moves: ",  this.constrainedMoves);
        utilities.Useful.printCollection("frozen moves: ",       this.frozenMoves);
        utilities.Useful.printCollection("work list moves: ",    this.workListMoves);
        utilities.Useful.printCollection("active moves: ",       this.activeMoves);
    }

    public void printInterferenceGraph(String fileName, String graphName) {
        GraphvizGen gg = new GraphvizGen(this.ai, this);
        try {
            gg.printInterferenceGraph(fileName, graphName);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public static void main(String a[]) throws IOException {
        if(a.length < 1) {
            System.err.println("java RegAlloc.IRC_Alloc file.dat");
            System.exit(1);
        } else {
            AllocationInstance ai = InKr.read(a[0]);
            IRC_Alloc alloc = new IRC_Alloc();
            alloc.setAllocationParameters(ai);
            alloc.performRegAllocation();
            alloc.setFinalLocation();
            DFA.drawControlFlowGraph(ai, "0", "fern.dot", true);
            OutKr ok = new OutKr(alloc, ai);
            ok.write();
        }
    }

}