package RegAlloc.ChordalAllocation;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;

import Graph.Node;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.Register;

/**
 * This class implements the post coalescing of nodes. The post coalescing is
 * a single stage in the register allocation process. After the nodes have
 * been allocated a color, the coalescer tries to make adjustments in the
 * interference graph, so that move related registers get the same color.
 */
public class PostNodeCoalescer {

    private int numberOfCoalescing = 0;

    private int numberOfMoveInstructions = 0;

    private int availableColors[] = null;

    private AllocationInstance ai = null;

    private HashMap<Register, Register> alias  = new HashMap<Register, Register>();

    private LinkedList<MovePair> movePairs = null;

    /**
     * Contructor method. The interference graph must have been already
     * colored.
     */
    public void initialize(AllocationInstance ai) {
        this.ai = ai;
        // check if all the nodes have been assigned a color:
        Collection<Node<Register>> c = this.ai.ig.getNodes();
        for(Node<Register> n : c) {
            if(n.getData().getColor() < 0) {
                System.err.println("Error: " + n.getData() + " hasn't been colored");
                System.exit(1);
            }
        }
        // create the collection of move pairs:
        this.movePairs = new LinkedList<MovePair>();
        this.numberOfMoveInstructions = 0;
        Collection<Instruction> moves = this.ai.iF.getMoveInstructions();
        for(Instruction move : moves) {
            Register src = this.getSingleElement(move.getDefSet());
            Register dst = this.getSingleElement(move.getUseSet());
            MovePair mp = new MovePair(src, dst);
            this.movePairs.add(mp);
            this.numberOfMoveInstructions++;
        }
        this.numberOfCoalescing = 0;
        this.availableColors = new int[this.ai.rF.getNumMachineRegisters()];
        this.alias = new HashMap<Register, Register>();
    }

    private <T> T getSingleElement(Collection<T> c) {
        for(T t : c) {
            return t;
        }
        return null;
    }


    /**
     * This method attempts to coalesce the pair of registers related by a move
     * instruction. The coalescing of two registers, t1 and t2 is performed by
     * greedy coloring. Greedy coloring is applied to the union of neighbors of t1
     * and t2. Notice that the original colors of t1 or t2 may be used in the
     * greedy coloring.
     */
    public void coalesce(AllocationInstance ai) {
        this.initialize(ai);
        while(movePairs.size() > 0) {
            MovePair mp = movePairs.removeFirst();
            this.coalescePair(mp);
        }
    }


    /**
     * This method coalesces two nodes. It chooses a color to the coalesced
     * nodes, which may not be the color of the original nodes. Also it
     * combine both nodes in a single one.
     */
    private void coalescePair(MovePair mp) {
        for(int i = 0; i < this.availableColors.length; i++)
            this.availableColors[i] = -1;
        Node<Register> src = this.ai.ig.getNode(mp.src);
        Node<Register> dst = this.ai.ig.getNode(mp.dst);
        boolean areTheSame = src.equals(dst);
        boolean areAdjacent = src.adj(dst);
        if(!areTheSame && !areAdjacent) {
            this.chooseCoalesceColor(src, dst);
            int choice = 0;
            if (mp.src instanceof MachineReg)
                choice++;
            if (mp.dst instanceof MachineReg)
                choice++;
            if(choice == 0) {
                for(int i = 0; i < this.availableColors.length; i++) {
                    if(this.availableColors[i] < 0) {
                        this.combine(src, dst, i);
                        this.numberOfCoalescing++;
                        break;
                    }
                }
            } else if (choice == 1) {
                if (mp.src instanceof MachineReg && this.availableColors[mp.src.getColor()] < 0) {
                    this.combine(src, dst, mp.src.getColor());
                    this.numberOfCoalescing++;
                } else if (mp.dst instanceof MachineReg && this.availableColors[mp.dst.getColor()] < 0) {               
                    this.combine(src, dst, mp.dst.getColor());
                    this.numberOfCoalescing++;
                }
            } else {
                System.err.println("Error: two interfering machine regs: " + mp.src.getName() + " and " + mp.dst.getName());
            }
        }
    }


    /**
     * This method chooses the most appropriate color for two nodes that are
     * going to be coalesced. It is possible that this color is not the
     * original color of any of the nodes.
     */
    private void chooseCoalesceColor(Node<Register> src, Node<Register> dst) {
        Collection<Node<Register>> j = src.neighbors();
        for(Node<Register> succ : j) {
            this.availableColors[succ.getData().getColor()] = 1; 
        }
        j = dst.neighbors();
        for(Node<Register> succ : j) {
            this.availableColors[succ.getData().getColor()] = 1;
        }
    }


    /**
     * This method merges two nodes in a single node. Given two nodes, say u
     * and v, it assigns every neighbor of v to u, removes v from the
     * interference graph, makes alias(v) = u, and makes sure that, for any
     * node w, if alias(w) = v (before), then alias(w) = u.
     */
    private void combine(Node<Register> u, Node<Register> v, int newColor) {
System.err.println("Combining " + u.getData() + " and " + v.getData() + " in color: " + newColor);
        u.linkAll(v.neighbors());
        u.getData().setColor(newColor);
        v.getData().setColor(newColor);
        this.ai.ig.remove(v.getData());
        this.updateAlias(v, u);
        this.updateMovePairs(u.getData(), v.getData());
    }


    /**
     * The alias table maintains is a mapping between the registers that have
     * been removed due to coalescing, and the registers that represent them
     * in the interference graph. When a calesced register is removed from the
     * interference graph, it is necessary to update this table. For instance,
     * assume that alias(u) = v, and v has been coalesced with w, so that
     * alias(v) = w. This method makes sure that alias(u) = w.
     */
    private void updateAlias(Node<Register> vNode, Node<Register> uNode) {
        Register v = vNode.getData();
        Register u = uNode.getData();
        this.alias.put(v, u);
        java.util.Set<Register> keys = this.alias.keySet();
        for(Register reg : keys) {
            Register actualValue = alias.get(reg);
            if(actualValue.equals(v)) {
                alias.put(reg, u);
                reg.setColor(u.getColor());
            }
        }
    }


    /**
     * After a register is coalesced, its name should not be referenced in any
     * part of the algorithm. This method replaces the name of the coalesced
     * register by the name of its alias in the move instructions.
     */
    private void updateMovePairs(Register newKey, Register oldKey) {
        for(int i = 0; i < this.movePairs.size(); i++) {
            MovePair mp = movePairs.get(i);
            if(mp.src.equals(oldKey))
                mp.src = newKey;
            else if(mp.dst.equals(oldKey))
                mp.dst = newKey;
        }
    }


    /**
     * This method returns the total number of move instructions found in this
     * interference graph.
     * @return an integer number.
     */
    public int getNumMoveInstructions() {
        return this.numberOfMoveInstructions;
    }


    /**
     * Informs how many times nodes have been combined during the execution of
     * the coalescing phase.
     * @return an integer number.
     */
    public int getNumCoalescedNodes() {
        return this.numberOfCoalescing;
    }


    /**
     * Return the mapping between coalesced nodes, and the nodes that
     * represent them in the final graph.
     * @return a hash mapping.
     */
    public HashMap<Register, Register> getCoalescedMapping() {
        return this.alias;
    }

}
