package RegAlloc.ChordalAllocation;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

import Graph.Node;
import RegAlloc.Register;

/**
 * This is a bucket list designed to map nodes to all the cliques in which they
 * are present. The main operation defined in this class is to remove a node
 * from the bucket. When a node is removed, it is also removed from every
 * other clique in which it is present. If due to this removal some of the
 * resulting cliques are smaller than the bucket threshold, such cliques are
 * also removed from the list, which is further updated.
 */
public class NodeBucket extends BucketList<CliquesPerNode> {

    private HashMap<Node<Register>, CliquesPerNode> nodes = null;

    public NodeBucket(int size) {
        super(size);
        this.nodes = new HashMap<Node<Register>, CliquesPerNode>();
    }

    public boolean add(CliquesPerNode cpn) {
        Node<Register> node = cpn.getNode();
        this.nodes.put(node, cpn);
        return super.add(cpn);
    }

    public void removeSmallCliques(Node<Register> node, int threshold) {
        CliquesPerNode cpn = this.nodes.get(node);
        if(cpn != null) {
            super.remove(cpn);
            LinkedList<Collection<Node<Register>>> listAux = new LinkedList<Collection<Node<Register>>>();
            Iterator<Collection<Node<Register>>> iter = cpn.iterator();
            while(iter.hasNext()) {
                Collection<Node<Register>> clique = iter.next();
                if(clique.size() > threshold)
                    listAux.add(clique);
            }
            CliquesPerNode newCpn = new CliquesPerNode(node);
            while(!listAux.isEmpty()) {
                Collection<Node<Register>> clique = listAux.removeFirst();
                newCpn.add(clique);
            }
        this.add(newCpn);
        }
    }

    public CliquesPerNode next() {
        LinkedList<CliquesPerNode> list = super.getTopList();
        CliquesPerNode bestClique = null;
        Register bestReg = null;
        for(CliquesPerNode cpn : list) {
            if(bestClique == null) {
                bestClique = cpn;
                bestReg = cpn.getNode().getData();
            } else {
                Node<Register> node = cpn.getNode();
                Register reg = node.getData();
                if(reg.spillingFactor() > bestReg.spillingFactor()) {
                    bestReg = reg;
                    bestClique = cpn;
                }
            }
        }
        list.remove(bestClique);
        return bestClique;
    }

}
