package RegAlloc.ChordalAllocation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import Graph.Graph;
import Graph.Node;
import RegAlloc.Register;

/**
 * This class represents a set of lists. Elements are distributed among the
 * lists based on a label. Elements with the same label are send to the same
 * list.
 */
public class LabeledListSet {

    private int highestLabel = 0;

    private ArrayList<HashSet<Node<Register>>> listSet = null;

    private int numElements = 0;

    public LabeledListSet(Graph<Register> graph) {
        this.highestLabel = 0;
        this.numElements = graph.size();
        this.listSet = new ArrayList<HashSet<Node<Register>>>(graph.size());
        for(int i = 0; i < graph.size(); i++)
            listSet.add(i, new HashSet<Node<Register>>());
        for(Node<Register> node : graph.getNodes() ) {
            Register cn = node.getData();
            cn.resetCardinality();
            cn.setStatus(Register.IN_LIST);
            listSet.get(0).add(node);
        }
    }


    /**
     * This method updates the label of the neighbors of the given parameter.
     * Let <I>i</I> be such a neighbor. If <I>i</I> has being already visited,
     * its label will be increased, and it will be moved to a list represented
     * by that label. If it hasn't being visited, it will be added to the list
     * with the label equals to 1.
     * @param node the node whose neighbors are going to be checked.
     */
    public void update(Node<Register> node) {
        updateCardTable(node.neighbors());
    }

    private void updateCardTable(Collection<Node<Register>> it) {
         for( Node<Register> node : it ) {
             Register cn = node.getData();
             switch(cn.getStatus()) {
                 case Register.REMOVED: break;
                 case Register.IN_LIST: {
                     int cardinality = cn.getCardinality();
                     listSet.get(cardinality).remove(node);
                     cn.incCardinality();
                     if( (cardinality + 1) > highestLabel )
                         highestLabel++;
                     listSet.get(cardinality + 1).add(node);
                     break;
                 }
                 default: break;
             }
         }
    }


    /**
     * Returns the next node available. Nodes are compared according to their
     * labels. The node with the highest label is always the next node to be
     * returned. If there are two or more nodes sharing the highest label, one
     * of them will be returned non-deterministically.
     * @return the node with the highest label.
     */
    public Node<Register> getNextNode() {
        if(highestLabel < 0)
            return null;
        while((listSet.get(highestLabel)).isEmpty()) {
            highestLabel--;
            if(highestLabel < 0)
                return null;
        }
        Node<Register> node = null;
        Iterator<Node<Register>> i = listSet.get(highestLabel).iterator();
        if(i.hasNext()) {
            node = i.next();
            Register cn = node.getData();
            cn.setStatus(Register.REMOVED);
            this.numElements--;
            i.remove();
        }
        return node;
    }


    /**
     * Returns the number of elements stored in this data structure.
     * @param an <CODE>int</CODE> number.
     */
    public int size() {
        return this.numElements;
    }
}
