package RegAlloc.ChordalAllocation;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.ListIterator;

import Digraph.DiNode;
import Graph.Node;
import Graph.SimpleGraph;
import Graph.SimpleNodeFc;
import RegAlloc.AllocationData;
import RegAlloc.AllocationInstance;
import RegAlloc.Instruction;
import RegAlloc.MachineReg;
import RegAlloc.MemoryAccessInstruction;
import RegAlloc.MemoryAccessRegister;
import RegAlloc.PseudoReg;
import RegAlloc.RegAllocator;
import RegAlloc.Register;
import RegAlloc.KrParser.InKr;
import RegAlloc.KrParser.OutKr;
import RegAlloc.RalfTools.DFA;
import RegAlloc.RalfTools.GraphvizGen;

/**
 * This class performs register allocation via coloring of chordal graphs.
 * In order to perform spills, it builds the list of maximal cliques of the
 * graph and removes registers until the interference graph can be colored with
 * the given number of colors. The name of the class stands for the spilling
 * technique been used: Destruction of Maximal Cliques.
 */
public class DMC_Alloc implements RegAllocator {

    private PostNodeCoalescer pnc = null;

    private int highestColor = 0;

    private boolean isKColorable = true;

    private boolean isChordal = true;

    private HashMap<MemoryAccessRegister, Register> spilledRegisterAlias  = new HashMap<MemoryAccessRegister, Register>();

    /**
     * This set contains all the registers removed in the spilling process.
     */
    private HashSet<Node<Register>> spilledNodes = new HashSet<Node<Register>>();

    /**
     * This variable contains all the registers spilled in the last iteration
     * of the spilling algorithm.
     */
    private HashSet<Register> lastSpills = new HashSet<Register>();

    /**
     * This variable is redundant, because every register in this set can also
     * be found in the spilledNodes set; however, it is used due to eficiency
     * reasons. Any register in this variable can be accessed in O(1).
     */
    private HashSet<Register> spilledRegisters = new HashSet<Register>();

    private AllocationInstance ai = null;

    /**
     * This method tries to find a coloring to the graph. Initially this method
     * thrown <CODE>nonChordalException</CODE> if the interference graph were not
     * chordal. Now it simply records this information for posterior use.
     */
    public void performRegAllocation() {
        System.err.println("Performing register allocation via choloring of chordal graphs. Spilling method: clique deconstruction.");
        int numColors = this.ai.rF.getNumMachineRegisters();
        ArrayList<MachineReg> regSet = null;
        this.isKColorable = false;

        do {
            this.updateControlFlowGraph();
            this.cleanControlFlowGraph();
            DFA.performLivenessAnalysis(this.ai, "0");
            this.buildInterferenceGraph();
            ChordalVerifier chordalVerifier = new ChordalVerifier(ai.ig);
            ListIterator<Node<Register>> vertexOrdering = chordalVerifier.getSimplicialEliminationOrdering();
            this.destroyMaximalCliques(vertexOrdering, numColors);
            regSet = this.applyColoring(vertexOrdering, numColors);
            this.allocatePseudos(regSet);
        } while (!this.isKColorable);

        this.coalesceRegisters();
        this.allocatePseudos(regSet);
        System.err.println("Done.");
    }


    private void cleanControlFlowGraph() {
        // reset the list of recent spills:
        this.lastSpills = new HashSet<Register>();
        // clean the control flow graph, eg.: remove the in and out set of all
        // the instruction:
        DFA.cleanControlFlowGraph(this.ai, "0");
    }


    /**
     * This method updates the control flow graph so that it can accommodate
     * spilled registers. Each spilled register will originate a new
     * instruction.
     */
    private void updateControlFlowGraph() {
        for(Register reg : this.lastSpills) {
            // place stores:
            Collection<Instruction> defInsts = reg.getDefInstructions();
            for(Instruction inst : defInsts) {
                // remove reg from inst
                inst.removeDef(reg);
                // creates a new instruction, where the new register will be inserted.
                MemoryAccessInstruction inst_store = this.ai.iF.createMemoryAccessInstruction(inst.getName());
                // create reg_store as a surrogate to the evicted register.
                MemoryAccessRegister reg_store = this.ai.rF.createMemoryAccessRegister(reg);
                inst.addDef(reg_store);
                reg_store.addDef(inst);
                inst_store.addUse(reg_store);
                reg_store.addUse(inst_store);
                // place the new instruction in the control flow graph:
                DiNode<Instruction> node = ai.cfg.getNode(inst);
                Collection<DiNode<Instruction>> succs = node.succs();
                Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
                for(DiNode<Instruction> succ : succs) {
                    toBeRemoved.add(succ);
                    succ.removePred(node);
                    ai.cfg.connect(inst_store, succ.getData());
                }
                for(DiNode<Instruction> succ : toBeRemoved) // avoid concurrent ex
                    node.removeSucc(succ);
                ai.cfg.connect(inst, inst_store);
                // make the alias: references to reg_store must be directed to reg
                this.spilledRegisterAlias.put(reg_store, reg);
            }

            // place loads:
            Collection<Instruction> useInsts = reg.getUseInstructions();
            for(Instruction inst : useInsts) {
                // remove reg from inst
                inst.removeUse(reg);
                // creates a new instruction, where the new register will be inserted.
                MemoryAccessInstruction inst_load = this.ai.iF.createMemoryAccessInstruction(inst.getName());
                // create reg_load as a surrogate to the loaded register.
                MemoryAccessRegister reg_load = this.ai.rF.createMemoryAccessRegister(reg);
                inst.addUse(reg_load);
                reg_load.addUse(inst);
                inst_load.addDef(reg_load);
                reg_load.addDef(inst_load);
                // place the new instruction in the control flow graph:
                DiNode<Instruction> node = ai.cfg.getNode(inst);
                Collection<DiNode<Instruction>> preds = node.preds();
                Collection<DiNode<Instruction>> toBeRemoved = new LinkedList<DiNode<Instruction>>();
                for(DiNode<Instruction> pred : preds) {
                    toBeRemoved.add(pred);
                    pred.removeSucc(node);
                    ai.cfg.connect(pred.getData(), inst_load);
                }
                for(DiNode<Instruction> pred : toBeRemoved) // avoid concurrent ex
                    node.removePred(pred);
                ai.cfg.connect(inst_load, inst);
                // make the alias: references to reg_load must be directed to reg
                this.spilledRegisterAlias.put(reg_load, reg);
            }
        }
    }


    /**
     * In order to perform the spilling of the least used color, the interference
     * graph is colored with many colors. The excess will be pruned by the
     * spilling phase.
     */
    private ArrayList<MachineReg> applyColoring(ListIterator<Node<Register>> regs, int numColors) {
        this.resetList(regs);

        // get the machine registers and color them
        ArrayList<MachineReg> mRegs = ai.rF.getMachineRegisters();
        for(int i = 0; i < mRegs.size(); i++)
            mRegs.get(i).setColor(i);

        // collor the pseudo registers.
        GreedyColoringAlgorithm gca = new GreedyColoringAlgorithm(numColors);
        this.highestColor = gca.color(regs);

        return mRegs;
    }


    /**
     * This method points the list iterator to the first position in the list.
     */
    public void resetList(ListIterator l) {
        while(l.hasPrevious()) {
            l.previous();
        }
    }


    /**
     * This method returns the number of colors used to color this graph. If the
     * graph is chordal, this number will be the size of the biggest clique, and
     * the graph will be chordal.
     * @return an integer value.
     */
    public int getNumberOfColors() {
        return this.highestColor + 1;
    }


    /**
     * Informs if the graph is chordal or not.
     * @return a boolean value that will be true if the graph is chordal, and
     * false otherwise.
     */
    public boolean isChordal() {
        return this.isChordal;
    }


    /**
     * This method spills registers in order to restrict the coloring to the
     * number of available colors. In order to remove nodes from the
     * interference graph, the algorithm lists the maximal cliques, and loops
     * removing nodes until there are no cliques bigger than the number of
     * available colors.
     * @param numAvailableColors the number of colors available. Nodes should
     * be spilled until the graph can be colored with the available colors.
     */
    private boolean destroyMaximalCliques(ListIterator<Node<Register>> nodes, int numColors) {
        this.resetList(nodes);
        BucketList<Collection<Node<Register>>> listOfCliques = this.buildListOfCliques(nodes, numColors);
        if(listOfCliques.size() > 0) {
            this.isKColorable = false;
            NodeBucket cliquesPerNodes = this.mapCliquesToNodes(listOfCliques);
            this.pruneNodes(cliquesPerNodes, listOfCliques, numColors);
            reconnectSpilledNodes();
        }
        else
            this.isKColorable = true;
        return this.isKColorable;
    }


    /**
     * This method removes nodes from the interference graph until it is K
     * colorable.
     * @param cliquesPerNodes a mapping m between nodes and cliques. If n is a
     * node, then m[n] = {c1, c2, ... cx}, where each ci contains n.
     * @param listOfCliques the list of all the cliques in the interference graph.
     * @param numColors the maximum number of colors available to color the
     * interference graph.
     */
    private void pruneNodes(NodeBucket cliquesPerNodes, BucketList<Collection<Node<Register>>> listOfCliques, int numColors) {
        while(listOfCliques.upperIndex() >= numColors) { // there are big cliques
            // select a node n to be removed:
            CliquesPerNode cpn = cliquesPerNodes.next();
            if(cpn == null)
                break;
            Node<Register> n = cpn.getNode();
            this.desconnectNode(n);
            // remove n from every clique in which it appears in cliquesPerNodes
            for(Collection<Node<Register>> clique : cpn) {
                listOfCliques.removeFromCollection(clique, cpn.getNode());
                if(clique.size() <= numColors) {
                    Iterator<Node<Register>> iter = clique.iterator();
                    while(iter.hasNext()) {
                        Node<Register> aux = iter.next();
                        cliquesPerNodes.removeSmallCliques(aux, numColors);
                    }
                }
            }
        }
    }


    /**
     * Each color is associated to a machine register. This method assigns to
     * each pseudo the machine register that is associated to the node's color.
     * Pay attention: THAT IS VERY IMPORTANT. If the machine registers are
     * colored before the greedy coloring is applied on the simplicial
     * elimination ordering, then the resulting coloring can not be total,
     * that is, maybe the number of colors will not be enough to color the
     * remaining pseudo registers. That is because the algorithm expects that
     * all the registers be passed in the simplicial order.
     */
    private void allocatePseudos(ArrayList<MachineReg> regSet) {
        ArrayList<PseudoReg> regs = ai.rF.getPseudoRegisters();
        for(Register reg : regs) {
            if(reg.getColor() < 0) {
                System.err.println("Spilled register: " + reg);
                if(!this.spilledRegisters.contains(reg)) {
                    Node<Register> node = this.ai.ig.getNode(reg);
                    this.desconnectNode(node);
                }
            }
            else if(reg.getColor() < regSet.size()) {
                reg.setAllocationDestiny(regSet.get(reg.getColor()));
            }
        }
    }


    private void desconnectNode(Node<Register> n) {
System.err.println("Spilling " + n.getData().getName());
        Collection<Node<Register>> j = n.neighbors();
        for(Node<Register> succ : j)
            succ.unlink(n);
        this.spilledNodes.add(n);
        this.spilledRegisters.add(n.getData());
        this.lastSpills.add(n.getData());
        this.isKColorable = false;
    }


    /**
     * This method builds a list with all the cliques whose size is bigger than
     * the number of registers available for allocation.
     */
    private BucketList<Collection<Node<Register>>> buildListOfCliques(ListIterator<Node<Register>> nodes, int numColors) {
        HashSet<Node<Register>> s = new HashSet<Node<Register>>();
        BucketList<Collection<Node<Register>>> list = new BucketList<Collection<Node<Register>>>(this.ai.ig.size(),  numColors);
        while(nodes.hasNext()) {
            Node<Register> node = nodes.next();
            Collection<Node<Register>> neighbors = node.neighbors();
            HashSet<Node<Register>> clique = new HashSet<Node<Register>>();
            clique.add(node);
            for(Node<Register> neighbor : neighbors)
                if(s.contains(neighbor)) {
                    clique.add(neighbor);
                }
            list.add(clique);
            s.add(node);
        }
        return list;
    }


    /**
     * This method gets a list of cliques, and builds a mapping m between the
     * nodes in the list, and the cliques. If n is a node, then
     * m[n] = {c1, c2, ... cx}, where each ci contains n. Only Pseudo Registers
     * that are not the source of a spill are part of the mapping.
     */
    private NodeBucket mapCliquesToNodes(BucketList<Collection<Node<Register>>> listOfCliques) {
        Collection<Node<Register>> nodeSet = this.ai.ig.getNodes();
        NodeBucket nodesToCliques = new NodeBucket(this.ai.ig.size());
        ArrayList<Collection<Node<Register>>> a = listOfCliques.getCollectionArray();
        for(Node<Register> node : nodeSet) {
            Register reg = node.getData();
            if(reg instanceof MachineReg || reg instanceof MemoryAccessRegister)
                continue;
            else {
                CliquesPerNode cpn = new CliquesPerNode(node);
                for(int j = 0; j < a.size(); j++)
                    if(a.get(j).contains(node)) {
                        cpn.add(a.get(j));
                    }
                nodesToCliques.add(cpn);
            }
        }
        if(nodesToCliques.size() == 0) {
            System.err.println("It is impossible to perform register allocation for this program.");
            System.exit(1);
        }
        return nodesToCliques;
    }


    /**
     * This method reconnect spilled nodes in order to compose a interference
     * graph containing only the spilled nodes. This graph allows to implement
     * aggressive coalescing among spilled nodes, so that move related nodes
     * can be mapped to the same memory location.
     */
    private void reconnectSpilledNodes() {
        for(Node<Register> n : this.spilledNodes) {
            Iterator<Node<Register>> j = n.neighbors().iterator();
            while(j.hasNext()) {
                Node<Register> succ = j.next();
                if(this.spilledNodes.contains(succ))
                    succ.link(n);
                else
                    j.remove();
            }
            this.ai.ig.remove(n.getData());
        }
    }


    /**
     * This method prints the interference graph formed by spilled nodes.
     */
    public void printSpilledGraph() {
        System.out.println("Spilled graph:");
        Iterator<Node<Register>> i = this.spilledNodes.iterator();
        while(i.hasNext()) {
            Node<Register> n = i.next();
            System.out.println(n);
        }
    }


    /**
     * This method attempts to coalesce the pair of registers related by a move
     * instruction. The coalescing of two registers, t1 and t2 is performed by
     * greedy coloring. Greedy coloring is applied to the union of neighbors of t1
     * and t2. Notice that the original colors of t1 or t2 may be used in the
     * greedy coloring.
     */
    public void coalesceRegisters() {
        this.pnc.coalesce(this.ai);
    }


    public Collection<Register> getAllocatedRegisters() {
        Collection<Node<Register>> c = this.ai.ig.getNodes();
        LinkedList<Register> l = new LinkedList<Register>();
        for(Node<Register> n : c) {
            if(!this.spilledNodes.contains(n)) {
                Register reg = n.getData();
                l.add(reg);
            }
        }
        java.util.Set<Register> colKeys = this.pnc.getCoalescedMapping().keySet();
        for(Register reg : colKeys) {
            l.add(reg);
        }
        return l;
    }


    public Collection<Register> getSpilledRegisters() {
        LinkedList<Register> l = new LinkedList<Register>();
        for(Node<Register> n : this.spilledNodes) {
            Register reg = n.getData();
            l.add(reg);
        }
        return l;
    }


    public int numMachineRegisters() {
        return this.ai.rF.getNumMachineRegisters();
    }


    public void checkAllocation(int originalSize) {
        int numberOfRegisters = this.ai.ig.size() + this.spilledNodes.size() + this.pnc.getNumCoalescedNodes();
        if (numberOfRegisters != originalSize)
            System.err.println("Invalid allocation. Original size = " + originalSize + ", current size: " + numberOfRegisters);
        else {
            Collection<Node<Register>> i = this.ai.ig.getNodes();
            for(Node<Register> n : i) {
                Collection<Node<Register>> j = n.neighbors();
                for(Node<Register> succ : j) {
                    if(n.getData().getColor() == succ.getData().getColor()) {
                        String s1 = n.getData().toString() + "(" + n.getData().getColor() + ")";
                        String s2 = succ.getData().toString() + "(" + succ.getData().getColor() + ")";
                        System.err.println("Invalid allocation: " + s1 + " = " + s2);
                    }
                }
            }
        }
    }


    /**
     * This method builds the interference graph from the liveness information
     * stored in the control flow graph.
     */
    private void buildInterferenceGraph() {
        this.ai.ig = new SimpleGraph<Register>();
        this.ai.ig.setNodeFactory(new SimpleNodeFc<Register>());

        // Add all the pseudo registers to the graph.
        ArrayList<PseudoReg> pseudoSet = this.ai.rF.getPseudoRegisters();
        for(PseudoReg pseudo : pseudoSet)
            if(!this.spilledRegisters.contains(pseudo))
                this.ai.ig.newNode(pseudo);

        // set the interferences between simultaneously alive registers
        Collection<DiNode<Instruction>> insts = this.ai.cfg.getNodes();
        for(DiNode<Instruction> nInst : insts) {
            ArrayList<Register> ll = nInst.getData().getInSet();
            for(int j = 0; j < ll.size(); j++) {
                Register r1 = ll.get(j);
                for(int k = j + 1; k < ll.size(); k++) {
                    Register r2 = ll.get(k);
                    this.ai.ig.connect(r1, r2);
                }
            }
        }

        // make the machine registers pairwise interfering
        ArrayList<MachineReg> regSet = this.ai.rF.getMachineRegisters();
        for(int i = 0; i < regSet.size() - 1; i++)
            for(int j = i+1; j < regSet.size(); j++)
                this.ai.ig.connect(regSet.get(i), regSet.get(j));

        // reset the cardinality of all the nodes
    }


    public void setAllocationParameters(AllocationInstance ai) {
        this.ai = ai;
        this.pnc = new PostNodeCoalescer();
    }


    public AllocationData getAllocationData() {
        int numRegisters = this.ai.ig.size() + this.spilledNodes.size();
        int numColors = this.highestColor + 1;
        int numSpills = this.spilledNodes.size();
        int numMoves = this.pnc.getNumMoveInstructions();
        int numCoalescings = this.pnc.getNumCoalescedNodes();
        boolean isChordal = this.isChordal;
        return new AllocationData(numRegisters, numColors, numSpills, numMoves, numCoalescings, isChordal);
    }


    /**
     * This method prints the numbers colected from the register allocation of the
     * given graph. The output format is given by: title, chordability, total number
     * of registers, number of colors used, number of spills, number of register
     * coalescing.
     * @param title the title of the graph
     * @param out the output chanel where the information should be printed.
     */
    public void printStatistics(String title, PrintStream out) {
        int numberOfRegisters = this.ai.ig.size() + this.spilledNodes.size();

        out.print(title + "\t\t " + this.isChordal + "\t\t " + numberOfRegisters + "\t\t ");
        out.print((this.highestColor + 1) + "\t\t " + this.spilledNodes.size() + "\t\t ");
        out.println(this.pnc.getNumCoalescedNodes() + "\t\t " + this.pnc.getNumMoveInstructions());
    }


    public static void main(String a[]) throws IOException {
        if(a.length < 1) {
            System.err.println("java TestAlg in.dat");
            System.exit(1);
        } else {
            AllocationInstance ai = InKr.read(a[0]);
            DMC_Alloc alloc = new DMC_Alloc();
            alloc.setAllocationParameters(ai);
            alloc.performRegAllocation();
            DFA.drawControlFlowGraph(ai, "0", "fern.dot", true);
            GraphvizGen gg = new GraphvizGen(ai, alloc);
            gg.toGraphviz("result.dot", "XXXXX");
            OutKr ok = new OutKr(alloc, ai);
            ok.write();
        }
    }

}