package RegAlloc.ChordalAllocation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.ListIterator;

import Graph.Graph;
import Graph.Node;
import RegAlloc.Register;

public class ChordalVerifier {

    private boolean isChordal = true;

    private Graph<Register> graph = null;

    private LinkedList<Node<Register>> vertexOrdering = null;

    private LinkedList<Node<Register>> nonSimplicialNodes = null;

    /**
     * Constructor method.
     * @param graph the graph that will be tested for chordality.
     */
    public ChordalVerifier(Graph<Register> graph) {
        this.graph = graph;
        this.nonSimplicialNodes = new LinkedList<Node<Register>>();
        this.isChordal = true;
    }


    /**
     * This method builds a simplicial elimination ordering for the graph
     * encapsulated by this object.
     * @return an object of the <CODE>ListIterator</CODE> type, that will
     * contain the elimination ordering found by the MCS algorithm. Notice
     * that if the graph is not chordal, than the list will not be a
     * simplicial elimination ordering.
     * @return a list of nodes.
     */
    public ListIterator<Node<Register>> getSimplicialEliminationOrdering() {
        if(this.vertexOrdering == null) {
            this.vertexOrdering = new LinkedList<Node<Register>>();
            LabeledListSet llSet = new LabeledListSet(graph);
            while(llSet.size() > 0) {
                Node<Register> node = llSet.getNextNode();
                vertexOrdering.add(node);
                llSet.update(node);
            }
        }
        return vertexOrdering.listIterator(0);
    }


    /**
     * Return a list with the non-simplicial nodes discovered during the
     * chordality test.
     * @return a list of nodes.
     */
    public ListIterator<Node<Register>> getNonSimplicialNodes() {
        return this.nonSimplicialNodes.listIterator(0);
    }


    /**
     * This method resets all the data previously stored in this object. It is
     * useful if, for example, the target graph has been changed. To save
     * processing, the chordality test is not redone when the methods
     * <CODE>getSimplicialEliminationOrdering</CODE> or <CODE>isChordal</CODE>
     * are invoked. After this method is called, all the tests will be
     * done again when those methods are invoked.
     */
    public void resetChordalTest() {
        this.isChordal = true;
        this.vertexOrdering = null;
        Collection<Node<Register>> c = this.graph.getNodes();
        for(Node<Register> n : c)
            n.getData().resetCardinality();
    }


    /**
     * This method verifies if a vertex is simplicial. A node is simplicial if
     * the graph contains an elimination ordering in which, when the node is
     * about to be eliminated, all its non-eliminated neighboors form a clique.
     * @param node the node that will be checked.
     * @return a boolean value that will be true if the given node is simplicial,
     * and false otherwise.
     */
    public static boolean isSimplicial(Node<Register> node) {
        ArrayList<Node<Register>> rep = new ArrayList<Node<Register>>(node.degree());
        Register cn = node.getData();
        cn.setStatus(Register.IN_LIST);
        for( Node<Register> neigh : node.neighbors() )
            if(neigh.getData().getStatus() == Register.IN_LIST)
                rep.add(neigh);
        for(Node<Register> nodeI : rep)
            for(Node<Register> nodeJ : rep)
                if( (nodeI != nodeJ) && !nodeI.adj(nodeJ))
                    return false;
        return true;
    }


    /**
     * This method informs if the graph is chordal or not. A chordal graph does
     * not have induced cycles with more than 4 vertices. That is, every cycle
     * with more than three vertices has a chord connecting non-adjacent
     * vertices.
     * @return a boolean value that will be true if the given graph is chordal,
     * and false otherwise.
     */
    public boolean isChordal() {
        if(this.vertexOrdering == null) {
            ListIterator<Node<Register>> li = this.getSimplicialEliminationOrdering();
            while(li.hasNext()) {
                Node<Register> node = li.next();
                if(!ChordalVerifier.isSimplicial(node)) {
                    this.nonSimplicialNodes.add(node);
                    System.out.println("Non-chordal node: " + node.getData());
                    this.isChordal = false;
                }
            }
        }
        return this.isChordal;
    }
}
