package RegAlloc.ChordalAllocation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * This class implements a list of buckets. Elements can be added to each
 * bucket, and one bucket can hold references to several entries. This class
 * is useful in algorithms that have to deal with ordered sets of limited
 * size. In this class, each bucket holds references to object of the
 * <CODE>java.util.Collection</CODE> type. The position of a given collection
 * in the list is determined by the size of the collection, that is, a
 * collection with 10 elements will be stored in the tenth bucket. Also it is
 * possible to determine a threshold. Collections with less elements than the
 * specified threshold will be not stored. Every time a collection is changed,
 * it is necessary to update its position in the list. The class have methods
 * to do this. Also, it implements a method to remove object of the collection.
 * When this method is used successfully, the collection is automatically
 * updated in the list of bucket.
 */
public class BucketList<E extends Collection> {

    private int upperIndex = 0;
    private int lowerIndex = 0;
    private int numElements = 0;
    private int threshold = 0;

    private ArrayList<LinkedList<E>> list = null;

    public BucketList(int size, int threshold) {
        list = new ArrayList<LinkedList<E>>(2*size);
        for(int i = 0; i < 2*size; i++)
            list.add(new LinkedList<E>());
        this.upperIndex = 0;
        this.lowerIndex = size;
        this.numElements = 0;
        this.threshold = threshold;
    }

    public BucketList(int size) {
        this(size, 0);
    }

    public boolean add(E c) {
        if(c.size() > threshold) {
            list.get(c.size()).add(c);
            if(this.upperIndex < c.size())
                this.upperIndex = c.size();
            if(this.lowerIndex > c.size())
                this.lowerIndex = c.size();
            this.numElements++;
            return true;
        } else
            return false;
    }

    public ArrayList<E> getCollectionArray() {
        ArrayList<E> c = new ArrayList<E>(this.numElements);
        for(int i = this.upperIndex; i >= this.lowerIndex; i--)
            if(!this.list.get(i).isEmpty()) {
                Iterator<E> iterator = list.get(i).iterator();
                while(iterator.hasNext())
                    c.add(iterator.next());
            }
        return c;
    }

    public void removeFromCollection(E c, Object o) {
        if(list.get(c.size()).contains(c)) {
            if(c.contains(o)) {
                list.get(c.size()).remove(c);
                c.remove(o);
                this.add(c);
                while(list.get(this.upperIndex).isEmpty() && this.upperIndex > 0)
                    this.upperIndex--;
            }
        }
    }

    public void update(E c, int oldSize) {
        if(list.get(oldSize).contains(c)) {
            list.get(oldSize).remove(c);
            this.add(c);
            while(list.get(this.upperIndex).isEmpty() && this.upperIndex > 0)
                this.upperIndex--;
        }
    }

    public boolean remove(E c) {
        if(c == null)
            return false;
        else {
            boolean retValue = list.get(c.size()).remove(c);
            while(list.get(this.upperIndex).isEmpty() && this.upperIndex != 0)
                this.upperIndex--;
            this.numElements--;
            return retValue;
        }
    }

    public E next() {
        while(list.get(this.upperIndex).isEmpty() && this.upperIndex != 0)
            this.upperIndex--;
        if(list.get(this.upperIndex).isEmpty())
            return null;
        E c = list.get(this.upperIndex).getFirst();
        this.remove(c);
        return c;
    }

    public LinkedList<E> getTopList() {
        while(list.get(this.upperIndex).isEmpty() && this.upperIndex != 0)
            this.upperIndex--;
        return list.get(this.upperIndex);
    }

    public int upperIndex() {
        return this.upperIndex;
    }

    public int lowerIndex() {
        return this.lowerIndex;
    }

    public int size() {
        return this.numElements;
    }

    public String toString() {
        String s = "";
        for(int i = this.threshold; i <= upperIndex; i++) {
            s += i + ": ";
            Iterator iterator = list.get(i).iterator();
            while(iterator.hasNext())
                s += iterator.next() + "; ";
            s += "\n";
        }
        return s;
    }
}
