package Graph;

import java.util.Collection;
import java.util.HashSet;

/**
 * This class implements a node of the graph library. It is a simple
 * implementation in which edges do not have direction.
 */
public class SimpleNode<E> implements Node<E> {

    private E e;

    private HashSet<Node<E>> neighbors = null;

    /**
     * Constructor method. It is made non-public because the implementation
     * of nodes should be restricted to the graph library. This constructor
     * will create a node whose identifier is the number 0.
     * @param e the information to be stored on this node.
     */
    SimpleNode(E e) {
        this.e = e;
        this.neighbors = new HashSet<Node<E>>();
    }


    /**
     * This method returns a reference to the node key.
     * @return The return value is a generic type. It will be the type of the
     * node key.
     */
    public E getData() {
        return e;
    }


    /**
     * This method inserts a new node among the list of successors of this
     * node.
     * @param node the successor node.
     */
    public boolean link(Node<E> node) {
        return this.neighbors.add(node);
    }


    /**
     * This method links the current node to all the nodes in the collection.
     * @param a collection of nodes. After this method is called, given any
     * node n in c, this.adj(c) == true;
     */
    public void linkAll(Collection<Node<E>> c) {
        for(Node<E> n : c) {
            this.neighbors.add(n);
            n.link(this);
        }
    }


    /**
     * This method removes a node from the list of successors of this node.
     * @param node the successor node.
     */
    public boolean unlink(Node node) {
        return this.neighbors.remove(node);
    }


    /**
     * This method removes all nodes from the list of successors of this node.
     */
    public void unlinkAll() {
        neighbors.clear();
    }


    /**
     * The next node after this, in the control flow graph.
     * @return a list of nodes, as an object of the <CODE>Collection<Node></CODE>
     * type. Notice that a node can have more than one successor.
     */
    public Collection<Node<E>> neighbors() {
        return neighbors;
    }


    /**
     * Number of nodes adjacent to this node in the control flow graph.
     * @return an integer that denotes the number of predecessors, and
     * successors of this node.
     */
    public int degree() {
        return neighbors.size();
    }


    /**
     * Informs if the given element is a key of one of the successors of this
     * node.
     * @param e a node key.
     * @return a <CODE>boolean</CODE> value, that will be true if the given
     * parameter is a successor of this node.
     */
    public boolean adj(Node<E> n) {
        if(neighbors.contains(n))
            return true;
        else
            return false;
    }


    public boolean equals(Object o) {
        return this.e.hashCode() == o.hashCode();
    }


    public int hashCode() {
        return this.e.hashCode();
    }


    /**
     * Textual representation of this node's key.
     * @return an object of the <CODE>String</CODE> type.
     */
    public String toString() {
        String s = this.e.toString();
        s += " -> ";
        for(Node<E> i : this.neighbors()) {
            s += i.getData().toString() + ", ";
        }
        return s;
    }
}
