package Graph;

import java.util.Collection;
import java.util.HashMap;

/**
 * This class represents an unoriented graph. This is a generic class, and the
 * type of the information stored in the graph's nodes is given by the
 * parametric data type E.
 */
public class SimpleGraph<E> implements Graph<E> {

    protected HashMap<E, Node<E>> graph = null;

    private NodeFactory<E> nFc = null;

    /**
     * This method removes the given node from the graph.
     * @param e the key of the node to be removed.
     * @return the removed node. If the node is not present in the graph, null
     * will be returned.
     */
    public Node<E> remove(E e) {
        Node<E> node = graph.remove(e);
        if(node != null) {
            Collection<Node<E>> c = node.neighbors();
            for(Node<E> neigh : c)
                neigh.unlink(node);
        }
        return node;
    }


    /**
     * This method determines if the two keys represent adjancent nodes in the
     * graph.
     * @param u one of the nodes
     * @param v the other node
     * @return true if the given keys are keys of adjacent nodes. False
     * otherwise.
     */
    public boolean adj(E u, E v) {
        Node<E> uNode = this.getNode(u);
        Node<E> vNode = this.getNode(v);
        if(vNode != null && uNode != null) {
            return uNode.adj(vNode) && vNode.adj(uNode);
        } else
            return false;
    }


    /**
     * Constructor method. Creates a new graph.
     */
    public SimpleGraph() {
        this(new SimpleNodeFc<E>());
    }


    /**
     * This method defines the factory of nodes that will be used in the
     * graph.
     * @param nFc the new factory of nodes.
     */
    public void setNodeFactory (NodeFactory<E> nFc) {
        this.nFc = nFc;
    }


    /**
     * Constructor method. Creates a new graph.
     */
    public SimpleGraph(NodeFactory<E> nFc) {
        this.nFc = nFc;
        graph = new HashMap<E, Node<E>>();
    }


    /**
     * This method creates a new node with the given key. After being created,
     * the new node is inserted into the graph.
     * @param e the internal data of the new node.
     * @return the newly created node.
     */
    public Node<E> newNode(E e) {
        if(!graph.containsKey(e)) {
            Node<E> n = nFc.create(e);
            graph.put(e, n);
            return n;
        } else {
            return graph.get(e);
        }
    }


    /**
     * Returns the number of nodes of this graph.
     * @return an integer number that denotes the number of nodes that are part
     * of this graph.
     */
    public int size() {
        return graph.size();
    }


    /**
     * This method adds a new edge to the graph, connecting the two keys
     * specified as parameters. This is a simple graph; therefore, adges
     * are inserted only between different nodes, that is, there is no
     * self loop.
     * @param f the node from which the edge will depart.
     * @param t the node to which the edge will go.
     */
    public void connect(E from, E to) {
        if(!this.graph.containsKey(from)) {
            this.newNode(from);
        }
        if(!this.graph.containsKey(to)) {
            this.newNode(to);
        }
        if(!from.equals(to)) {
            Node<E> t1 = this.getNode(from);
            Node<E> t2 = this.getNode(to);
            t1.link(t2);
            t2.link(t1);
        }
    }


    public Node<E> getNode(E e) {
        return this.graph.get(e);
    }


    /**
     * This method returns a list of the nodes stored in the graph.
     * @return an object of the <CODE>Collection</CODE> type.
     */
    public Collection<Node<E>> getNodes() {
        return graph.values();
    }


    /**
     * This method determines if the graph is a clique. A clique is a complete
     * graph.
     * @return a boolean value.
     */
    public boolean isClique() {
        Collection<Node<E>> i = graph.values();
        Collection<Node<E>> j = graph.values();
        for(Node<E> x: i)
            for(Node<E> y: j)
                if(!x.adj(y)) return false;
        return true;
    }


    public boolean contains(E e) {
        return graph.containsKey(e);
    }


    /**
     * Returns a textual representation of this graph.
     * @return an object of the <CODE>String</CODE> type.
     */
    public String toString() {
        String s = "";
        Collection<Node<E>> c = graph.values();
        for(Object e: c) {
            s += "--------------------------\n" + e.toString() + '\n';
        }
        return s;
    }

}
