package Graph;

import java.util.Collection;

/**
 * This class represents an unoriented graph. This is a generic class, and the
 * type of the information stored in the graph's nodes is given by the
 * parametric data type E.
 */
public interface Graph<E> {

    /**
     * This method defines the factory of nodes that will be used in the
     * graph.
     * @param nFc the new factory of nodes.
     */
    public void setNodeFactory (NodeFactory<E> nFc);


    /**
     * This method creates a new node with the given key. After being created,
     * the new node is inserted into the graph.
     * @param e the internal data of the new node.
     * @return the newly created node.
     */
    public Node<E> newNode(E e);


    /**
     * Returns the number of nodes of this graph.
     * @return an integer number that denotes the number of nodes that are part
     * of this graph.
     */
    public int size();


    /**
     * This method adds a new edge to the graph, connecting the two keys
     * specified as parameters. This is a simple graph; therefore, adges
     * are inserted only between different nodes, that is, there is no
     * self loop.
     * @param f the node from which the edge will depart.
     * @param t the node to which the edge will go.
     */
    public void connect(E from, E to);


    /**
     * This method returns the node that is associated to the given
     * key.
     * @param e the key of the node that will be returned.
     * @return the node associated to the given key, if it is present in the
     * graph, and null otherwise.
     */
    public Node<E> getNode(E e);


    /**
     * This method determines if the two keys represent adjancent nodes in the
     * graph.
     * @param u one of the nodes
     * @param v the other node
     * @return true if the given keys are keys of adjacent nodes. False
     * otherwise.
     */
    public boolean adj(E u, E v);


    /**
     * This method returns a list of the nodes stored in the graph.
     * @return an object of the <CODE>Collection</CODE> type.
     */
    public Collection<Node<E>> getNodes();


    /**
     * Informs if the graph contains the given node.
     * @return a boolean value that will be true if the node is present in the
     * graph, and will be false otherwise.
     */
    public boolean contains(E e);


    /**
     * This method removes the given node from the graph.
     * @param e the key of the node to be removed.
     * @return the removed node. If the node is not present in the graph, null
     * will be returned.
     */
    public Node<E> remove(E e);


    /**
     * This method determines if the graph is a clique. A clique is a complete
     * graph.
     */
    public boolean isClique();
}
