package Digraph;

import java.util.HashMap;
import java.util.Collection;

/**
 * This class represents an oriented graph. This is a generic class, and the
 * type of the information stored in the graph's nodes is given by the
 * parametric data type E.
 */
public class SimpleDigraph<E> implements Digraph<E> {

    protected HashMap<E, DiNode<E>> graph = null;

    private DiNodeFactory<E> nFc = null;

    /**
     * This method removes the given node from the graph.
     * @param e the key of the node to be removed.
     * @return the removed node. If the node is not present in the graph, null
     * will be returned.
     */
    public DiNode<E> remove(E e) {
        DiNode<E> node = graph.remove(e);
        if(node != null) {
            Collection<DiNode<E>> succs = node.succs();
            for(DiNode<E> succ : succs)
                succ.removePred(node);
            Collection<DiNode<E>> preds = node.preds();
            for(DiNode<E> pred : preds)
                pred.removeSucc(node);
        }
        return node;
    }


    /**
     * Constructor method. Creates a new graph.
     */
    public SimpleDigraph() {
        this(new DiNodeFc<E>());
    }


    /**
     * This method defines the factory of nodes that will be used in the
     * graph.
     * @param nFc the new factory of nodes.
     */
    public void setNodeFactory (DiNodeFactory<E> nFc) {
        this.nFc = nFc;
    }


    /**
     * Constructor method. Creates a new graph.
     */
    public SimpleDigraph(DiNodeFactory<E> nFc) {
        this.nFc = nFc;
        graph = new HashMap<E, DiNode<E>>();
    }


    /**
     * This method creates a new node with the given key. After being created,
     * the new node is inserted into the graph.
     * @param e the internal data of the new node.
     * @return the newly created node.
     */
    public DiNode<E> newNode(E e) {
        if(!graph.containsKey(e)) {
            DiNode<E> n = nFc.create(e);
            graph.put(e, n);
            return n;
        } else
            return graph.get(e);
    }


    /**
     * Returns the number of nodes of this graph.
     * @return an integer number that denotes the number of nodes that are part
     * of this graph.
     */
    public int size() {
        return graph.size();
    }


    /**
     * This method adds a new edge to the graph, connecting the two keys
     * specified as parameters. This is a simple graph; therefore, adges
     * are inserted only between different nodes, that is, there is no
     * self loop.
     * @param f the node from which the edge will depart.
     * @param t the node to which the edge will go.
     */
    public void connect(E from, E to) {
        if(!this.graph.containsKey(from)) {
            this.newNode(from);
        }
        if(!this.graph.containsKey(to)) {
            this.newNode(to);
        }
        if(!from.equals(to)) {
            DiNode<E> t1 = this.getNode(from);
            DiNode<E> t2 = this.getNode(to);
            t1.addSucc(t2);
            t2.addPred(t1);
        }
    }


    public boolean goesTo(E origin, E destiny) {
        if( !this.graph.containsKey(origin) || !this.graph.containsKey(destiny)) {
            return false;
        } else {
            DiNode<E> t1 = this.getNode(origin);
            DiNode<E> t2 = this.getNode(destiny);
            return t1.isPred(t2);
        }
    }

    public DiNode<E> getNode(E e) {
        return this.graph.get(e);
    }


    /**
     * This method returns a list of the nodes stored in the graph.
     * @return an object of the <CODE>Collection</CODE> type.
     */
    public Collection<DiNode<E>> getNodes() {
        return graph.values();
    }


    /**
     * This method determines if the graph is a clique. A clique is a complete
     * graph.
     * @return a boolean value.
     */
    public boolean isClique() {
        Collection<DiNode<E>> i = graph.values();
        Collection<DiNode<E>> j = graph.values();
        for(DiNode<E> x: i)
            for(DiNode<E> y: j)
                if(!x.isPred(y) || !x.isSucc(y)) return false;
        return true;
    }


    public boolean contains(E e) {
        return graph.containsKey(e);
    }


    /**
     * Returns a textual representation of this graph.
     * @return an object of the <CODE>String</CODE> type.
     */
    public String toString() {
        String s = "";
        Collection<DiNode<E>> c = graph.values();
        for(DiNode<E> n : c) {
            String line = n.getData().toString() + '\n';
            s += line;
            Collection<DiNode<E>> succs = n.succs();
            for(DiNode<E> succ : succs) {
                s += " goes to " + succ.getData().toString() + " ";
            }
            s += "\n";
        }
        return s;
    }

}
