package Digraph;

import java.util.Collection;

/**
 * This class represents an oriented graph. This is a generic class, and the
 * type of the information stored in the graph's nodes is given by the
 * parametric data type E.
 */
public interface Digraph<E> {

    /**
     * This method defines the factory of nodes that will be used in the
     * graph.
     * @param nFc the new factory of nodes.
     */
    public void setNodeFactory (DiNodeFactory<E> nFc);


    /**
     * This method creates a new node with the given key. After being created,
     * the new node is inserted into the graph.
     * @param e the internal data of the new node.
     * @return the newly created node.
     */
    public DiNode<E> newNode(E e);


    /**
     * Returns the number of nodes of this graph.
     * @return an integer number that denotes the number of nodes that are part
     * of this graph.
     */
    public int size();


    /**
     * This method adds a new edge to the graph, connecting the two keys
     * specified as parameters. This is a simple graph; therefore, adges
     * are inserted only between different nodes, that is, there is no
     * self loop.
     * @param f the node from which the edge will depart.
     * @param t the node to which the edge will go.
     */
    public void connect(E from, E to);


    /**
     * Returns the node that is wrapping the key e, if present.
     * @param e the searching key.
     * @return an object of the <CODE>DiNode<E></CODE> type.
     */
    public DiNode<E> getNode(E e);


    /**
     * Informs if the node origin is a predecessor of node destiny.
     * @param origin the possible predecessor.
     * @param destiny the possible successor.
     * @return a boolean value that is true if origin node is a predecessor of
     * destiny node; false otherwise.
     */
    public boolean goesTo(E origin, E destiny);


    /**
     * This method returns a list of the nodes stored in the graph.
     * @return an object of the <CODE>Collection</CODE> type.
     */
    public Collection<DiNode<E>> getNodes();


    /**
     * This method determines if the graph is a clique. A clique is a complete
     * graph.
     * @return a boolean value.
     */
    public boolean isClique();

    /**
     * This method removes the given node from the graph.
     * @param e the key of the node to be removed.
     * @return the removed node. If the node is not present in the graph, null
     * will be returned.
     */
    public DiNode<E> remove(E e);


    /**
     * Reports if the graph contains the given key.
     * @param e the search key.
     * @return true if there is a node in the graph containing the search key.
     * False otherwise.
     */
    public boolean contains(E e);


    /**
     * Returns a textual representation of this graph.
     * @return an object of the <CODE>String</CODE> type.
     */
    public String toString();

}
