package rme.security;

import arcademis.Stub;
import arcademis.Stream;
import arcademis.Invoker;
import arcademis.OrbAccessor;
import arcademis.RemoteReference;
import arcademis.MarshalException;
import arcademis.NetworkException;
import arcademis.security.Key;
import arcademis.security.Certificate;
import arcademis.security.ArcademisCryptoException;

/**
 * This stub has a list of servers that it uses in order to choose the best one
 * to perform a given remote call.
 */
public class SecureStub extends Stub {

    private Invoker invoker = OrbAccessor.getInvoker();

    private Stream certificate;

    private Key prvKey = null;

    /**
     * This method binds to this stub the certificate that will be used to
     * validate remote calls. This certificate has been encrypted by the
     * discovery agency.
     * @param stream the encrypted certificate.
     */
    public void setCertificate(Stream stream) {
        this.certificate = stream;
    }

    /**
     * Returns the certificate used by this stub. The certificate is a
     * stream of encrypted bytes.
     * @return the encrypted certificate
     */
    public Stream getCertificate() {
        return this.certificate;
    }

    /**
     * The stub needs to know the private key of the client, because the client
     * have to decrypt messages sent to him by the service provider. Also, the
     * certificate the client receives from the discovery agency comes
     * encrypted with the client's public key. This certificate is deciphered
     * in this method's body.
     * @param prvKey the private key the client uses.
     */
    public void setOwner(Key prvKey) throws ArcademisCryptoException {
        this.prvKey = prvKey;
        prvKey.decrypt(this.certificate);
        try {
            this.certificate = (Stream)this.certificate.readObject();
        } catch (MarshalException me) {
            me.printStackTrace();
        }
    }

    /**
     * This method is responsible for creating a call with the given parameters
     * and passing it to the server side of a distributed application.
     * @param args the arguments of the call.
     * @param opCode the code of the operation that is being invoked. In order
     * to allow the communication between the stub and the skeleton, any
     * remote operation is given a particular code.
     * @servers a <CODE>char</CODE> that describes the pattern for server
     * invocation. This character may hold three different service combinators:
     * sequential invocation (>), non-deterministic invocation (?) and concurrent
     * invocation (|).
     * @return NetworkException in case some error occurs during the invocation
     * or remote processing of the call.
     * @throws NetworkException if something goes wrong during the remote
     * invocation.
     */
    public Stream invoke(Stream args, int opCode, char serviceCombinator, int priority)
    throws NetworkException {
        RemoteReference rRef = this.mr.nextReference();
        SecureRemoteCall remoteCall = new SecureRemoteCall(rRef.getIdentifier(), args, opCode);
        remoteCall.setCertificate(this.certificate);
        return invoker.invoke(remoteCall, rRef);
    }

    /** 
     * Fills the stream b with the byte sequence that describes this object.
     * @throws MarshalException if it is not possible to serialize this object.
     * @param the stream used in the serialization process.
     */
    public void marshal(Stream b) throws MarshalException {
        b.write(this.certificate);
        super.marshal(b);
    }

    /**
     * Fills the content of this object with information retrived from a
     * stream.
     * @param the stream used in the serialization process.
     */     
    public void unmarshal(Stream b) throws MarshalException {
        this.certificate = (Stream)b.readObject();
        super.unmarshal(b);
    }
}
