package rme.security;

import arcademis.*;
import rme.rmep.*;

/**
 * Synchronous request sender that implements the best effort semantics. This
 * implementation does not create any new thread, being executed in the same
 * thread of the calling application.
 * <P>
 * The SecureRequestSender implementation stores the communication channel
 * because it can be reused in another method invocation. The RME stub
 * tries to recycle the communication channel.
 */
public class SecureRequestSender extends RequestSender {

	/**
	 * The remoteCall object stores the parameter of the call and, after its
	 * execution, it also holds the return value.
	 */
	private SecureRemoteCall remoteCall = null;

	private Stream future = null;

    /**
     * This request sender is really simple. It just sends the message that has
     * been associated to it in the stub by means of the protocol instance that
     * also has been associated with it in the implementation of
     * <CODE>RmeStub</CODE>.
     * @param ch the channel implementation used to transmit the message.
     */
    public void open(Channel ch) {
        try {
            super.protocol.setChannel(ch);

            remoteCall.setReturnAddress(ch.getLocalEpid());
            long randValidator = this.randLong();
            remoteCall.setRandomValidator(randValidator);

            // client -{SecureCall}-> server
            SecureCallMsg callMsg = (SecureCallMsg)OrbAccessor.getMessage(SecureConstants.SECURE_CALL_MESSAGE);
            callMsg.setRemoteCall(remoteCall);
            super.protocol.send(callMsg);

            // client <-{RQuestion}- server
            Message msg1 = super.protocol.recv();
            if(!(msg1 instanceof RQuestionMsg)) {
                String unknowClassName = msg1.getClass().getName();
                    throw new ProtocolException("Wrong message type: " + unknowClassName);
            }

            // client -{RReplay}-> server
            RReplayMsg rMsg = (RReplayMsg)OrbAccessor.getMessage(SecureConstants.R_REPLAY_MESSAGE);
            rMsg.setRandomValidator(((RQuestionMsg)msg1).getRandomValidator() - 1);
            super.protocol.send(rMsg);

            // client <-{RetMsg}- server
            Message msg2 = super.protocol.recv();
            if(!(msg2 instanceof RetMsg)) {
                String unknowClassName = msg2.getClass().getName();
                throw new ProtocolException("Wrong message type: " + unknowClassName);
            }

            this.future = ((RetMsg)msg2).getReturnValue();

        } catch (ArcademisException e) {
            e.printStackTrace();
        }
    }

	/**
	 * Determines the remote call to be performed.
	 * @param remoteCall the remote call that will be performed by this
	 * request sender.
	 */
	public void setRemoteCall(SecureRemoteCall remoteCall) {
		this.remoteCall = remoteCall;
	}

	/**
	 * Returns the remote call that is hold by this request sender.
	 * @return an object of the <CODE>RemoteCall</CODE> type.
	 */
	public RemoteCall getRemoteCall() {
		return this.remoteCall;
	}

	/**
	 * Returns the communication channel that is hold by this request sender.
	 * @return an object of the <CODE>Channel</CODE> type.
	 */
	public Channel getChannel() {
		return super.protocol.getChannel();
	}

	/**
	 * Return the valued obtained as the result of the remote call.
	 * @return an object of the <CODE>Stream</CODE> type.
	 */
	public Stream getFuture() {
		return this.future;
	}


        private long randLong() {
                return (long) (java.lang.Math.random()*Long.MAX_VALUE);
        }

}
