package rme.security;

import arcademis.*;
import arcademis.concreteComponents.*;
import rme.RmeRemoteCall;
import arcademis.security.Certificate;

/**
 * This class contains all the parameters of a remote call and the return value
 * originated from its processing. All remote calls contains a reference for
 * the remote entity that is responsible for processing it. They also contains
 * a list of arguments and a reference for the return value generated from its
 * processing. Further parameters can be added in the extension of this class.
 */
public class SecureRemoteCall extends RmeRemoteCall {

    private Stream certificate;

    private long randValue = 0L;

    /**
     * The empty constructor. It is necessary because all marshalable objects
     * in Arcademis, needs one. The serialization protocol asks for a default
     * empty constructor to read an object from a byte sequence.
     */
    public SecureRemoteCall() {}

    /**
     * Constructor method. It sets the call parameters and creates a unique
     * identifier for the call.
     * @param destiny the remote reference to the object that will process this
     * call.
     * @param arguments the arguments of the call.
     */
    public SecureRemoteCall(Identifier targetObjectIdentifier, Stream arguments, int opCode) {
        super(targetObjectIdentifier, arguments, opCode);
        this.randValue = 0L;
    }

    /**
     * This method binds to this stub the certificate that will be used to
     * validate remote calls. This certificate has been encrypted by the
     * discovery agency.
     * @param stream the encrypted certificate.
     */
    public void setCertificate(Stream stream) {
        this.certificate = stream;
    }

    /**
     * Returns the certificate used by this stub. The certificate is a 
     * stream of encrypted bytes.
     * @return the encrypted certificate
     */
    public Stream getCertificate() {
        return this.certificate;
    }


    /**
     * Every secure message carries a random number. This number helps to
     * avoid replay attacks. Upon receiving a secure message, the peer can
     * write a confirmation to the sender, with the encrypted random
     * number. If the sender ever responds, it is because it actually
     * wants to contact the receiver.
     * @param randValue the number that will be used to mark this message.
     */
    public void setRandomValidator(long randValue) {
        this.randValue = randValue;
    }

    /**
     * Returns the random number used to validate this message.
     * @return a <CODE>long</CODE> value.
     */
    public long getRandomValidator() {
        return this.randValue;
    }


    /**
     * Gives textual information about this remote call.
     * @return an <CODE>String</CODE> representing the internal state of this
     * remote call.
     */
    public String toString() {
        return super.toString() +
        "\nRandom validator: " + this.randValue +
        "\nCertificate: " + this.certificate;
    }

    /**
     * Fills the stream b with the byte sequence that describes this object.
     * @throws MarshalException if it is not possible to serialize this object.
     * @param the stream used in the serialization process.
     */
    public void marshal(Stream b) throws MarshalException {
        // send the random validator
        b.write(this.randValue);
        // send the certificate
        b.write(this.certificate);
        super.marshal(b);
    }

    /**
     * Fills the content of this object with information retrived from a
     * stream.
     * @param the stream used in the serialization process.
     */
    public void unmarshal(Stream b) throws MarshalException {
        this.randValue = b.readLong();
        this.certificate = (Stream)b.readObject();
        super.unmarshal(b);
    }

    private Certificate plain = null;

    /**
     * This method only exist to allow the network layer to send the
     * certificate to the dispather. Actually, the cipher could have
     * being decripted by the dispacher, but the network layer needs
     * the plain certificate to confirm the client is really sending
     * the message, and it is not a man-in-the-middle attack.
     */
    public void setPlainCertificate(Certificate plain) {
        this.plain = plain;
    }

    /**
     * Returns the certificate used by this stub. The certificate is a 
     * stream of encrypted bytes.
     * @return the encrypted certificate
     */
    public Certificate getPlainCertificate() {
        return this.plain;
    }

}
