package rme.security;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Date;

import arcademis.Stream;
import arcademis.RemoteCall;
import arcademis.OrbAccessor;
import arcademis.MarshalException;
import arcademis.ArcademisException;
import arcademis.server.Dispatcher;
import arcademis.security.Key;
import arcademis.security.Certificate;
import arcademis.server.DispatcherDecorator;
import arcademis.security.ArcademisCryptoException;

import rme.SNService.KeyManager;

public class SecureDispatcher extends DispatcherDecorator {

    private Key pubKey = null;

    private ArrayList methods = null;

    private boolean[] access = null;

    public SecureDispatcher(Dispatcher dispatcher) {
        super(dispatcher);
    }


    public Stream dispatch(RemoteCall c) {
        SecureRemoteCall src = (SecureRemoteCall)c;
        int opCode = src.getOperationCode();
        if(this.access[opCode] == false) {
            try {
                String name = this.checkSignature(src);
                this.checkPermission(name, opCode);
            } catch (ArcademisException e) {
                return reportError(e);
            }
        }
        return super.dispatcher.dispatch(c);
    }


    /**
     * Fills a stream with an exception, so it can be sent back to the client.
     */
    private Stream reportError(ArcademisException e) {
        Stream returnValue = OrbAccessor.getStream();
        try {
            returnValue.write((Exception)e);
        } catch (MarshalException me) {
            me.printStackTrace();
        }
        return returnValue;
    }


    /**
     * This method verifies if the given name has permission to access the
     * remote method
     */
    public void checkPermission(String name, int opCode) throws ArcademisCryptoException {
        HashSet hs = (HashSet)this.methods.get(opCode);
        if(!hs.contains(name)) {
            System.err.println(name + " attempted to execute operation " + opCode + " without permission.");
            throw new ArcademisCryptoException();
        }
    }


    /**
     * Verifies the name server signature
     */
    private String checkSignature(SecureRemoteCall s) throws ArcademisCryptoException {
        Certificate certificate = null;

        // deciphering the discovery service signature
        try {
            Stream stream = s.getCertificate();
            this.pubKey.verify(stream);
            certificate = (Certificate)stream.readObject();
        } catch (Exception e) {
            throw new ArcademisCryptoException("Invalid signature");
        }

        // checking the validity of the key:
        Date timeStamp = certificate.getTimeStamp();
        Date presentTime = new Date();
        if(timeStamp.getTime() + KeyManager.VALIDITY < presentTime.getTime())
            throw new ArcademisCryptoException("Invalid signature");

        return certificate.getName();
    }

    /**
     * This method defines the number of remote methods implemented by
     * the skeleton which encapsulates this dispatcher.
     */
    public void setNumberOfMethods(int numOfMethods) {
        this.methods = new ArrayList(numOfMethods);
        this.access = new boolean[numOfMethods];
        for(int i = 0; i < numOfMethods; i++) {
            this.methods.add(i, new HashSet());
            this.access[i] = true;
        }
    }


    /**
     * Binds the public key used by the discovery agency to this dispatcher.
     * This key will be used to validate the certificates issued by the
     * discovery agency.
     * @param key the public key of the discovery agency.
     */
    public void setLookupServicePublicKey(Key key) {
        this.pubKey = key;
    }


    /**
     * Returns to the caller the public key the dispacher is using to verify
     * signatures. This is (or should be) the public key generated by the
     * discovery agency. The corresponding private key is used by the discovery
     * agency to issue certificates.
     * @return the public key used to verify messages.
     */
    public Key getLookupServicePublicKey() {
        return this.pubKey;
    }


    /**
     * Grants execution of a single method to a single user.
     */
    public void grantPermission(int method, String name) {
        HashSet temp = (HashSet)this.methods.get(method);
        temp.add(name);
    }

    /**
     * Revoke all execution permissions on a single method. This routine is
     * mostly used before invoking the <CODE>grantPermission</CODE> method a
     * number of times in order to define a small set of users that are
     * allowed to execute an specific method.
     */
    public void revokeAllPermissions(int method) {
        HashSet temp = (HashSet)this.methods.get(method);
        temp.clear();
        this.access[method] = false;
    }

    public void revokeAllPermissions() {
        for(int i = 0; i < this.methods.size(); i++)
            this.access[i] = false;
    }

    /**
     * Grants free access to every method.
     */
    public void grantFreeAccess() {
        for(int i = 0; i < this.methods.size(); i++)
            this.access[i] = true;
    }

    /**
     * Allows a single method to be freely executed by any user.
     */
    public void grantFreeAccess(int method) {
        this.access[method] = true;
    }

    /**
     * Allows a single user to execute any method.
     */
    public void grantFreeAccess(String name) {
        for(int i = 0; i < methods.size(); i++) {
            HashSet temp = (HashSet)this.methods.get(i);
            temp.add(name);
        }
    }

}
