package rme.rmep;

import arcademis.*;
import arcademis.server.*;
import arcademis.concreteComponents.*;
import arcademis.security.ArcademisCryptoException;
import rme.server.*;
import rme.RmeConstants;
import rme.security.SecureRemoteCall;
import rme.security.SecureConstants;

/**
 * This class represents the call message, one of the four kinds of messages used
 * in RMEP. The call message describes a message containing all the values need
 * in order to carry on a remote method invocation.
 */
public class SecureCallMsg implements Message {

    private SecureRemoteCall remoteCall = null;

    /**
     * Determines the remote call paramteres that will be transmited by means
     * of the message.
     * @param remoteCall an object that encapsulates all the parameters of a
     * remote call.
     */
    public void setRemoteCall(SecureRemoteCall remoteCall) {
        this.remoteCall = remoteCall;
    }

    /**
     * Gives a reference to the remote call that is or was transmited by this
     * message.
     * @return an object of the <CODE>SecureRemoteCall</CODE> type.
     */
    public SecureRemoteCall getRemoteCall() {
        return this.remoteCall;
    }

    /**
     * Fills the stream b with the byte sequence that describes this object.
     * @throws MarshalException if it is not possible to serialize this object.
     * @param the stream used in the serialization process.
     */
    public void marshal(Stream b) throws MarshalException {
        b.write(this.remoteCall);
    }

    /**
     * Fills the content of this object with information retrived from a
     * stream.
     * @param the stream used in the serialization process.
     */
    public void unmarshal(Stream b) throws MarshalException {
        this.remoteCall = (SecureRemoteCall)b.readObject();
    }


    /**
     * Passes the remote call to the dispatcher and sends the result back to the
     * client.
     * @param the request receiver that received this message.
     */
    public void execute(ProtocolHandler s) throws ArcademisException {
        // First, check if the sender is actually who he says he is.
        // server -{RQuestion}-> client
        RQuestionMsg conf = (RQuestionMsg)OrbAccessor.getMessage(SecureConstants.R_QUESTION_MESSAGE);
        conf.setRandomValidator(this.remoteCall.getRandomValidator());
        s.getProtocol().send(conf);

        // server <-{RReplay}- client
        RReplayMsg replay = (RReplayMsg)s.getProtocol().recv();
        if(!conf.validate(replay)) {
            System.err.println("Random validators do not match.");
            System.err.println("Expected validator: " + conf.getRandomValidator());
            System.err.println("Received validator: " + replay.getRandomValidator());
            throw new ArcademisCryptoException("Possible replay attack: random numbers do not match.");
        }

        // executes the call, and send results back to client
        Dispatcher dispatcher = ((RequestReceiver)s).getDispatcher();
        Stream returnValue = dispatcher.dispatch(remoteCall);

        // server -{RetMsg}-> client
        ((RmeRequestReceiver)s).sendReturnValue(remoteCall.getCallIdentifier(), returnValue, s.getProtocol());
    }

}
