/*
 * This code has been adapted from the original Ninja RMI stub generator.
 * The author of Arcademis and RME is Fernando Pereira. The author of
 * NinjaRMI is Matt Welsh.
 * The original copyrith is as follows:
 *
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

package rme.rmec;

import java.lang.reflect.*;
import java.io.*;
import java.util.Vector;
import java.util.HashSet;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;


/**
 * <P>
 * RMECompiler is a class which generates Java sourcecode implementing
 * RME stubs and skeletons, given a class which implements interfaces
 * extending <tt>arcademis.Remote</tt>. I've left the methods public here
 * in case you come up with a cunning way to use RMECompiler within another
 * program. The usual way to use this class is to invoke
 * <tt>java rme.rmec.RmeC</tt> which is a <tt>main</tt> front-end to it.
 * </P>
 * <P>
 * This class contains methods for generating stubs and skeletons. It
 * generates code for the RME platform. This class can be easily
 * modified in order to generate different kinds of stubs and skeletons.
 * Comments along the code make this task easier.
 * </P>
 * <P>
 * If necessary to change the code generated for the stub, the method
 * <CODE>writeStubOutput</CODE> must be modified. If necessary to change
 * the skeleton code, modify the method <CODE>writeSkeletonOutput</CODE>. 
 * </P>
 */
public class RMECompiler implements StubSkeletonGenerator {

    /**
     The target class, i.e. the class from which the stub and the skeleton
     will be generated.
     */
    private Class cl;

    /**
     * This field holds information about the target class structure, such
     * as methods, superclasses, implemented interfaces, etc.
     */
    private ClassDecomposer cldc;

    /**
     * The name of the stub and skeleton classes respectivaly.
     */
    private String stubname, skelname;

    /**
     * The remote methods that will be part of the stub and the skeleton.
     */
    private HashSet allMethods;

    /**
     * This vector contais a reference to all that the remote interfaces the
     * target class implements. Code will be generated for each of the
     * methods declared in these interfaces.
     */
    private Vector allintfs;

    /**
     * The short name of the target class. The short name is given by the
     * cannonical name without the package description.
     */
    private String class_shortname;

    /**
     * The name of the package that contains the target class. It may be null.
     */
    private String packagename;


    /**
     * This method parses the name of the target class, and splits it into
     * the package name, and the short name of the class. For example, for
     * <CODE>java.lang.String</CODE>, the package name is <CODE>java.lang</CODE>,
     * and the short class name is <CODE>String</CODE>.
     * @param classname is the name of the target class. A stub and a skeleton
     * will be generated for this class.
     */
    private void parseClassName(String className) {
        StringTokenizer st = new StringTokenizer(className, ".");
        boolean first = true;
        while (st.hasMoreTokens()) {
            String t = st.nextToken();
            if (st.hasMoreTokens()) {
                if (!first) {
                    packagename = packagename + "." + t;
                } else {
	            packagename = t;
                    first = false;
                }
            } else {
                class_shortname = t;
            }
        }
        this.stubname = class_shortname+"_Stub";
        this.skelname = class_shortname+"_Skeleton";
    }


    /**
     * This method verifies if the target class implements at least one
     * <CODE>arcademis.Remote</CODE> interface. If it does, then all the
     * remote interfaces it implements are stored in the global array
     * <CODE>allintfs</CODE>.
     * @param classname is the name of the target class. A stub and a skeleton
     * will be generated for this class.
     * @throws IllegalArgumentException if the class does not implement any
     * remote interface.
     */
    private void searchForRemoteInterfaces(String className) throws IllegalArgumentException, ClassNotFoundException {

        this.cl = Class.forName(className);
        this.allintfs = new Vector();
        Class interfaces[] = cl.getInterfaces();
        boolean foundRemoteInterface = false;

        for (int i = 0; i < interfaces.length; i++) {
            Class theints[] = interfaces[i].getInterfaces();
            if (theints != null) {
                for (int j = 0; j < theints.length; j++) {
                    if (theints[j].getName().equals("arcademis.Remote")) {
                        this.allintfs.addElement(interfaces[i]);
                        foundRemoteInterface = true;
                        break;
                    }
                }
            }
        }

        if(!foundRemoteInterface)
            throw new IllegalArgumentException("Class " + className + " does not implement any interfaces which extend arcademis.Remote.");
    }


    /**
     * This method saves all the methods declared in the remote interfaces
     * implemented by the target class. Only public methods are saved. The
     * saved methods will be used in the generation of code for the stub and
     * the skeleton.
     */
    private void findRemoteMethods() throws IllegalArgumentException {
        this.allMethods = new HashSet();

        for(int i = 0; i < allintfs.size(); i++) {
            Class interf = (Class)allintfs.get(i);
            ClassDecomposer decomp = new ClassDecomposer(interf);
            MethodDecomposer md = decomp.getNextMethod();
            while(md != null) {
                if(!this.isValidMethod(md))
                    continue;
                allMethods.add(md);
                md = decomp.getNextMethod();
            }
        }
    }


   /**
    * Create an RMECompiler for the given class.
    * @param classname is the name of the target class. A stub and a skeleton
    * will be generated by this class. The stub/skeleton pair will contain
    * code to handle all the remote methods implemented by the given class.
    * This means that every method declared in a remote interface that is
    * implemented by the given class will appear in the generated code.
    * If the class is part of a package, its full name must be given to the
    * stub/skeleton generator, e.g.: <CODE>java.lang.String</CODE>.
    */
    public RMECompiler(String className) throws IllegalArgumentException, ClassNotFoundException {
        this.parseClassName(className);
        this.searchForRemoteInterfaces(className);
        this.findRemoteMethods();
    }


    /**
     * Returns the name of the generated stub class.
     * @return an object of the <CODE>String</CODE> time.
     */
    public String getStubName() {
      return stubname;
    }

    /**
     * Returns the name of the generated skeleton class.
     * @return an object of the <CODE>String</CODE> time.
     */
    public String getSkelName() {
      return skelname;
    }

    /**
     * Returns the name of the package which contains the target class.
     * For instance, if target is <CODE>java.lang.String</CODE>, the
     * package <CODE>java.lang</CODE> will be returned.
     * @return an object of the <CODE>String</CODE> time.
     */
    public String getPackageName() {
      return packagename;
    }

    /**
     * Returns the name of the target class, without the package prefix.
     * For instance, if target is <CODE>java.lang.String</CODE>, the
     * name <CODE>String</CODE> will be returned.
     * @return an object of the <CODE>String</CODE> time.
     */
    public String getClassShortName() {
      return class_shortname;
    }


    // This section of the code deals exclusively with the generation of the stub

    /**
     * This method generates the stub code. If necessary to change the stub
     * implementation, it is this method that should be modified. The body of this
     * method is divided into four main components:
     * <UL>
     * <LI> <CODE>writeStubHeader</CODE>: writes any header information the
     * middleware developer think is necessary. Examples are the package name,
     * the import statements, etc.
     *
     * <LI> <CODE>writeStubClassDesc</CODE>: writes the declaration of the
     * class that implements the stub. This declaration includes the class
     * name, all the extended classes, and all the implemented interfaces.
     *
     * <LI> <CODE>writeStubMethods</CODE>: writes all the stub methods. A new
     * method will be generated for every remote method the target class
     * implements.
     *
     * <LI> <CODE>writeStubClosing</CODE>: this part writes to the generated
     * class any information the middleware developer wants to apper in the
     * class file. Examples are additional methods in the stub class,
     * comments and inner classes.
     * </UL>
     * @param os the file that will receive the generated output.
     */
    public void writeStubOutput(PrintWriter os) throws IllegalArgumentException {
      writeStubHeader(os);
      writeStubClassDesc(os);
      writeStubMethods(os);
      writeStubClosing(os);
    }

    /**
     * Change this method if necessary to change the list of imported packages,
     * or if necessary to change the package name of the generated class.
     */
    private void writeStubHeader(PrintWriter os) {
        os.println("/* Generated by RMECompiler.java -- do not edit */");
        if (packagename != null) {
            os.println("\npackage "+packagename+";");
        }
        os.println("\nimport rme.*;");
        os.println("import arcademis.*;\n");
    }


    /**
     * Change this method if necessary to change the superclass of the stubs.
     * This method can also me mofified when necessary to make the stub code
     * to implement extra interfaces.
     */
    private void writeStubClassDesc(PrintWriter os) {
        String superClassName = "MultiServerStub";
        os.print("public class " + stubname + " extends " + superClassName + " implements ");
        for (int i = 0; i < allintfs.size(); i++) {
            os.print(((Class)allintfs.elementAt(i)).getName());
            if (i != allintfs.size() - 1) {
                os.print(", ");
            }
        }
        os.println(" {");
    }


    /**
     * If necessary to add extra functions in the generated code, then change
     * this method. The method <CODE>writeStubHeader</CODE> can also be
     * altered with the same purpose. In this case, the auxiliary methods
     * will be listed before the remote methods.
     */
    private void writeStubClosing(PrintWriter os) {
        os.println("  // End of generated code");
        os.print("}");
    }


    /**
     * The stub class contains the implementation of all the methods
     * declared in the remote interfaces the target class implements.
     * The code for these methods in generated by this method. If
     * necessary to change the implementation of the remote methods,
     * then this method (<CODE>writeStubMethods</CODE>) should be
     * changed.
     */
    private void writeStubMethods(PrintWriter os) throws IllegalArgumentException {

        Iterator methIter = allMethods.iterator();
        for (int methodnum = 0; methIter.hasNext(); methodnum++) {

            MethodDecomposer md = (MethodDecomposer)methIter.next();
            Method m = md.getMethod();
            TypeDecomposer td = new TypeDecomposer(m.getReturnType());

            // prints method declaration, e.g: public void m(int p1, ...) {
            os.print("  public "+td.toString()+" "+m.getName());
            os.println(md.getParamsDesc() + md.getThrowsDesc() + " {");

            // the generated code is slightly different, depending on the return
            // type of the method being void or not. The initializer routing
            // loads an initial value in the stream that will receive the return
            // value.
            boolean returnIsVoid = m.getReturnType().getName().equals("void");
            if (!returnIsVoid)
                os.println("    " +td.toString()+ " resp = " +initializer(m.getReturnType())+ ";");
            else
                os.println("    Object resp = " +initializer(m.getReturnType())+ ";");

            os.println("    try {");

            // prints code to store the arguments in the stream 'args'. This stream
            // will be sent to the skeleton in the other side of the network.
            os.println("      Stream args = OrbAccessor.getStream();");
            ParameterDecomposer params[] = md.getParams();
            if (params != null)
                if(params.length > 0)
                    for (int i = 0; i < params.length; i++)
                        os.println("      args.write("+params[i].getName()+");");

            os.println("      int op = " + methodnum + ";");
            os.println("      Stream future = invoke(args, op, '?', 0);");
            os.println("      if(future.isException()) {");
            os.println("        Exception e = (Exception)future.readObject();");
            os.println("        throw e;");
            os.println("      }");

            // again, two different codes, depending on the return value.
            // In RME, the return type will
            // be loaded in a stream called future. After the remote execution, such
            // stream must be read, and, depending on the return type, a type cast
            // may be necessary.
            if (!returnIsVoid) {
                td = new TypeDecomposer(m.getReturnType());
                os.println("      resp = (" +td.toString()+ ")future." +objinread(m.getReturnType())+ "();");
            } else {
                os.println("      resp = future.readObject();");
            }

            // write routines for treating the exceptions that can be thrown by this method
            String ex[] = getExceptionNames(m);
            for(int x = 0; x < ex.length; x++) {
                os.println("    } catch (" + ex[x]+ " e) {");
                os.println("      throw e;");
            }
            if (!throwsRemoteException(m, "java.lang.Exception")) {
                os.println("    } catch (Exception e) {");
                os.println("      throw new arcademis.UnspecifiedException(e.toString());");
            }
            os.println("    }");

            if (!returnIsVoid)
                os.println("    return resp;");

            os.println("  }\n");
        }
    }


    // Skeleton methods *******************************************************

    /**
     * This method generates the skeleton code. If necessary to change the skeleton
     * implementation, it is this method that should be modified. The body of this
     * method is divided into four main components, in the same way as the
     * <CODE>writeStubOutput</CODE> method. For more details, refer to that
     * method's code and comments.
     */
    public void writeSkelOutput(PrintWriter os) throws IllegalArgumentException {
        writeSkelHeader(os);
        writeSkelClassDesc(os);
        writeSkelMethods(os);
        writeSkelClosing(os);
    }


    /**
     * Change this method if necessary to change the list of imported packages,
     * or if necessary to change the package name of the generated class.
     */
    private void writeSkelHeader(PrintWriter os) {
        os.println("/* Generated by RMECompiler.java -- do not edit */");
        if (packagename != null) {
            os.println("\npackage "+packagename+";");
        }
        os.println("\nimport rme.*;");
        os.println("import rme.server.*;\n");
        os.println("import arcademis.*;\n");
    }


    /**
     * Change this method if necessary to change the superclass of the skeleton.
     * This method can also me mofified when necessary to make the skeleton
     * to implement extra interfaces.
     */
    private void writeSkelClassDesc(PrintWriter os) {
        os.println("public class "+skelname+" extends arcademis.server.Skeleton {");
    }


    /**
     * If necessary to add extra functions in the generated code, then change
     * this method. The method <CODE>writeSkeletonHeader</CODE> can also be
     * altered with the same purpose. In this case, the auxiliary methods
     * will be listed before the <CODE>dispatch</CODE> method.
     */
    private void writeSkelClosing(PrintWriter os) {
        os.println("}");
    }


    /**
     * In RME, the skeleton has just one method, called <CODE>dispatch()</CODE>.
     * The body of this method is a big switch, in which each entry is a
     * different remote method. If necessary to change the implementation of
     * the <CODE>dispatch()</CODE> method, than the alterations should be
     * performed in the code below.
     */
    private void writeSkelMethods(PrintWriter os) throws IllegalArgumentException {
        os.println("\n  public Stream dispatch(RemoteCall r) throws Exception {");
        os.println("\n    RmeRemoteCall remoteCall = (RmeRemoteCall)r;");
        os.println("    Stream returnStr = OrbAccessor.getStream();");
        os.println("    Stream args = remoteCall.getArguments();\n");
        os.println("    switch (remoteCall.getOperationCode()) {");

        Iterator methIter = allMethods.iterator();
        for (int methodnum = 0; methIter.hasNext(); methodnum++) {
            MethodDecomposer md = (MethodDecomposer)methIter.next();
            Method m = md.getMethod();

            os.println("      case "+methodnum+": {");

            // reads the arguments from the stream, one at a time:
            ParameterDecomposer params[] = md.getParams();
            if (params != null) {
                for (int i = 0; i < params.length; i++) {
                    os.println("        "+params[i].getTypeName()+" "+params[i].getName()+" = (" +params[i].getTypeName()+ ")args."+objinread(params[i].getType())+"();");
                }
            }

            // as in the stub case, the skeleton code may be different, depending on
            // the type of return.
            TypeDecomposer td = new TypeDecomposer(m.getReturnType());
            if (!m.getReturnType().getName().equals("void")) {
                os.print("        "+td.toString()+" retValue = ");
            } else {
                os.println("        Marshalable retValue = null;");
                os.print("        ");
            }

            // pass the arguments to the remote object implementation.
            // E.g: int retValue = ((RemoteObjClassName)super.remoteObject).meth(param0, param1);
            os.print("((" +cl.getName()+ ")super.remoteObject)."+m.getName()+"(");
            if (params != null) {
                for (int i = 0; i < params.length; i++) {
                    os.print(params[i].getName());
                    if (i != params.length-1) os.print(", ");
                }
            }
            os.println(");");

            os.println("        returnStr.write(retValue);");
            os.println("      }");
            os.println("      break;");
        }

        // if the received method does not match any of the methods the skeleton
        // implements, an error must be thrown.
        os.println("      default: {");
        os.println("        throw new ArcademisException(\"Invalid operation in remote method request\");");
        os.println("      }");
        os.println("    }    // end switch");
        os.println("\n    return returnStr;");
        os.println("  }    // end dispatch\n");
    }


    // The remaining program contain auxiliary functions


    /**
     * This method determines if the target method is valid or not. Only valid
     * methods must appear in the code of stubs and skeletons. In the RME system,
     * a method is valid if it is public, is part of a remote interface, and
     * throws <CODE>ArcademisException</CODE>. This method does not verifies if
     * the target method is declared in a remote interface, because it assumes
     * that such step has been performed already.
     * @param m the target method.
     */
    private boolean isValidMethod(MethodDecomposer md) throws IllegalArgumentException {

        Method m = md.getMethod();

        /*
          Checks if the method throws arcademis.ArcademisException. Every remote
          method must throw this exception, because it is the way the middleware
          have to report a failure due to the distributed nature of the system
          to the application developer.
        */
        if (!throwsRemoteException(m, "arcademis.ArcademisException"))
            throw new IllegalArgumentException("Method "+m.getName()+" must throw arcademis.ArcademisException.");
        /*
          If the method is not public, it should be not part of the stub,
          because it cannot be invoked on the target object.
        */
        int methmods = m.getModifiers();
        if (!Modifier.isPublic(methmods))
            return false;
        else
            return true;
    }


    /**
     * This auxiliary method checks if the target method throws a given
     * exception.
     */
    private boolean throwsRemoteException(Method m, String e) {
        Class ex[] = m.getExceptionTypes();
        int i;
        boolean found = false;
        for (i = 0; i < ex.length; i++) {
            if (ex[i].getName().equals(e)) {
	        found = true; break;
            }
        }
        return found;
    }

    // return an array with all the exceptions that can be risen by the method
    private String[] getExceptionNames(Method m) {
        Class ex[] = m.getExceptionTypes();
        sort(ex);
        String list[] = new String[ex.length];

        for (int i = 0, j = 0; i < ex.length; i++) {
            list[j++] = ex[i].getName();
        }
        return list;
    }

    // order a vector of classes so that the most general classes stay in higher
    // positions in the vector
    private void sort(Class v[]) {
        for(int i = 0; i < v.length - 1; i++) {
            int mark = i;
            for(int j = i+1; j < v.length; j++) {
                if(v[mark].isAssignableFrom(v[j])) {
                    mark = j;
                }
            }
            Class aux = v[mark];
            v[mark] = v[i];
            v[i] = aux;
        }
    }

    // Return true if methods have same name and signature
    private boolean methodsEqual(MethodDecomposer m1, MethodDecomposer m2) {
        return m1.equals(m2);
    }

    /**
    Given a type, determine the proper value to initialize one of its
    instances. This method is used in the generation of code for
    stubs, in order to initialize the variable that will hold the
    return value. For instance, int n = 0; double d = 0.0;
    boolean b = false; etc
    */
    private String initializer(Class type) {
        if (type.isAssignableFrom(Boolean.TYPE)) return "false";
        if (type.isAssignableFrom(Byte.TYPE)) return "0";
        if (type.isAssignableFrom(Character.TYPE)) return "0";
        if (type.isAssignableFrom(Double.TYPE)) return "0.0";
        if (type.isAssignableFrom(Float.TYPE)) return "0.0";
        if (type.isAssignableFrom(Integer.TYPE)) return "0";
        if (type.isAssignableFrom(Long.TYPE)) return "0L";
        if (type.isAssignableFrom(Short.TYPE)) return "0";
        return "null";
    }

    /**
    Given a class, determine the appropriate method of
    <CODE>RmeStream</CODE>
    to read it. This method is necessary because arguments sent
    accross the network are transmited as raw sequences of bytes. Upon
    arrival, it is necessary to recover the orignal value from the
    stream. It is used in the generation of code for the skeleton and
    for the stub. For example, if necessary to read an integer
    number, write <CODE>int param0 = (int)args.readInt()</CODE>;
    */
    private String objinread(Class type) {
        if (type.isAssignableFrom(Boolean.TYPE)) return "readBoolean";
        if (type.isAssignableFrom(Byte.TYPE)) return "readByte";
        if (type.isAssignableFrom(Character.TYPE)) return "readChar";
        if (type.isAssignableFrom(Double.TYPE)) return "readDouble";
        if (type.isAssignableFrom(Float.TYPE)) return "readFloat";
        if (type.isAssignableFrom(Integer.TYPE)) return "readInt";
        if (type.isAssignableFrom(Long.TYPE)) return "readLong";
        if (type.isAssignableFrom(Short.TYPE)) return "readShort";
        return "readObject";
    }

}
