package rme.SNService;

import java.util.Hashtable;
import java.util.Enumeration;

import arcademis.security.Key;
import arcademis.security.Signature;

/**
 * In addition of imposing time constraints on issued certificates, the
 * name server also revokes the registration of clients after 24 hours. This
 * obliges clients to be constantly updating their cryptographic keys.
 * The purpose of this class is to delete keys whose valid time has expired.
 * The class has an internal thread (just one) that wakes in regular periods
 * of time to check the validity of every registered key. If the key is not
 * more valid, it removes it from the symbol table.
 */
public class KeyManager extends Hashtable implements Runnable {

    /**
     * This constant determines the validity period of a key. The discovery
     * agency stores keys for at most 24 hours. After this period, the client
     * must update its key. Because normally remote invocations are short, the
     * client do not need to use the same key for a long time; therefore, the
     * 24 hours life suits the necessities of many applications.
     */
    public static final int VALIDITY = 86400000;

    /**
     * The key manager is a thread the awakes periodically in order to remove
     * keys whose validity has expired. This constant defines the period the
     * key manager wakes.
     */
    public static final int CHECK_TIME = 60000;


    // time that a key is valid for a client
    private long maxTime;  
    //frequency of the key validity check
    private long check_period;

    private Thread t = null;
    
    public KeyManager() {
    	maxTime = VALIDITY;
    	check_period = CHECK_TIME;
    	
    	//runs a background thread to check expirations
    	t = new Thread(this);
    	t.start();
    }

    public KeyManager(long mtime, long p) {
    	maxTime = mtime;
    	check_period = p;
    	
    	//runs a background thread to check expirations
    	t = new Thread(this);
    	t.start();
    }

    /**
     * The method that runs when the thread starts.  It sleeps and wakes up
     * periodically to check on the expiration of all the keys in the hashtable
     */
     public void run() {
     		while(true) {
     			//sleep
			try {
     	 			t.sleep(check_period);
			} catch (Exception e) {
				e.printStackTrace();
			}
     	 		
     	 		//go through all the keys
     	 		long currentTime = System.currentTimeMillis();
     	 		Enumeration enum = super.keys();
        		while(enum.hasMoreElements()) {
        	    		Signature name = (Signature)enum.nextElement();
            			KeyData data = (KeyData)super.get(name);
            			long insertTime = (long)data.getInsertTime();
   	         		//if max time has elapsed, delete
        	    		if (insertTime + maxTime < currentTime) {
            				Object o = super.remove(name);
System.err.println("KeyManager, removing: " + o);
            			}		
        		}
     		}			
     }
     
     
    /**
     * Every key is bound to a validity time. This period of time is
     * determined by the put method. After this time has ellapsed, the key
     * must be removed from the catalogue.
     */
     public Object put(Object signature, Object pubKey) {
     		if ( (signature instanceof Signature) && (pubKey instanceof Key) ) {
     				KeyData data = new KeyData (pubKey, System.currentTimeMillis());
     				return super.put (signature, data);
     			
     		} else {
            System.err.println("Incompatible types in key binding: " + signature.getClass().getName() + ", " + pubKey.getClass().getName());
            return null;
        }
    }


    public Object get(Object signature) {
        KeyData data = (KeyData)super.get(signature);
        if(data != null)
            return data.getPubKey();
        else
            return null;
    }
    

    void printKeyCatalogue() {
        System.out.println("----------------------------------------");
        Enumeration enum = super.keys();
        while(enum.hasMoreElements()) {
            Signature name = (Signature)enum.nextElement();
            //Key key = (Key)super.get(name);
            KeyData data = (KeyData)super.get(name);
            Key key = (Key)data.getPubKey();
            System.out.println(name + " --> " + key);
        }
    }

}
