package arcademis.server;

import arcademis.RemoteCall;
import arcademis.Stream;

/**
 * The dispatcher's decorator permits to aggregate extra functionalities into
 * the core dispatcher component. Such capabilities include, for example, log
 * generation and security policies. Besides adding more capabilities to the
 * dispatcher, this decorator also allows to change some aspects of the server
 * structure, such as the thread policy, for example.
 */
public class DispatcherDecorator extends Dispatcher {

    protected Dispatcher dispatcher = null;

    /**
     * Constructor method of the dispatcher decorator. It associates the
     * decorated dispatcher with this one.
     * @param dispatcher the decorated dispatcher.
     */
    public DispatcherDecorator(Dispatcher dispatcher) {
        this.dispatcher = dispatcher;
    }

    /**
     * This method simply repasses to the decorated dispatcher the received
     * parameter. In order to add further funcitonality to the dispatcher, this
     * method can be overwritten in order to perform the extra processsing.
     * @param c the remote call to be processed.
     * @return a stream holding the return value of the call.
     */
    public Stream dispatch(RemoteCall c) {
        return this.dispatcher.dispatch(c);
    }

    /**
     * This method is used for reification purposes. Whenever necessary to
     * travel accross the chain of decorators it may be used.
     */
    public Dispatcher getDecoratedDispatcher() {
        return this.dispatcher;
    }

    public Skeleton getSkeleton() {
        return this.dispatcher.getSkeleton();
    }

    public void setSkeleton(Skeleton skel) {
        this.dispatcher.setSkeleton(skel);
    }

    /**
     * This method defines the scheduler bound to this dispatcher.
     * @param scheduler an object of the <CODE>Scheduler</CODE> type.
     */
    public void setScheduler(Scheduler scheduler) {
        this.dispatcher.setScheduler(scheduler);
    }


    /**
     * Returns a reference to the scheduler bound to this dispatcher.
     * @return an object of the <CODE>Scheduler</CODE> type.
     */
    public Scheduler getScheduler() {
        return this.dispatcher.getScheduler();
    }
}
