package arcademis.security;

import java.io.Serializable;
import arcademis.Stream;
import arcademis.MarshalException;

public class XorKey implements Key, Serializable {

    private static boolean DEBUG_MODE = false;

    private byte SECRET_KEY = 73;

    private boolean isPublic = false;

    public XorKey() {
        this.DEBUG_MODE = false;
    }

    /**
     * This method receives a stream of bytes and sign it. The ciphered
     * data will replace the original data in the stream. The signature is
     * always performed with the private key. In order to recover the signed
     * data, it is necessary to use the <CODE>verify</CODE> method.
     * @param stream the object that holds the data to be encrypted. Notice
     * that the original data stored in this stream will be lost.
     * @throws ArcademisCryptoException if this is not a private key.
     */
    public void sign(Stream stream) throws ArcademisCryptoException {
        if(!DEBUG_MODE) {
            byte[] plain = stream.getBytes();
            stream.clean();
            for(int i = 0; i < plain.length; i++)
                plain[i] = (byte)(plain[i] ^ SECRET_KEY);
            stream.fill(plain);
        }
    }


    /**
     * This method verifiers a signed stream. To verify an stream means to
     * decipher with the public key the sequence of bytes that was encrypted
     * with the corresponding private key. The same stream given as
     * a parameter is used to hold the decrypted data. Only public keys
     * can be used to verify an stream.
     * @param stream the stream to be decrypted. The original contents of
     * this stream will be lost.
     * @throws ArcademisCryptoException if this is not a public key.
     */
    public void verify(Stream stream)  throws ArcademisCryptoException {
        if(!DEBUG_MODE) {
            byte[] plain = stream.getBytes();
            stream.clean();
            for(int i = 0; i < plain.length; i++)
                plain[i] = (byte)(plain[i] ^ SECRET_KEY);
            stream.fill(plain);
        }
    }


    /**
     * This method receives a stream of bytes and encrypts it. The ciphered
     * data will replace the original data in the stream.
     * @param stream the object that holds the data to be encrypted. Notice
     * that the original data stored in this stream will be lost.
     */
    public void encrypt(Stream stream) throws ArcademisCryptoException {
        if(!DEBUG_MODE) {
            byte[] plain = stream.getBytes();
            stream.clean();
            for(int i = 0; i < plain.length; i++)
                plain[i] = (byte)(plain[i] ^ SECRET_KEY);
            stream.fill(plain);
        }
    }


    /**
     * This method deciphers an encrypted stream. The same stream given as
     * a parameter is used to hold the decrypted data.
     * @param stream the stream to be decrypted. The original contents of
     * this stream will be lost.
     */
    public void decrypt(Stream stream)  throws ArcademisCryptoException {
        if(!DEBUG_MODE) {
            byte[] plain = stream.getBytes();
            stream.clean();
            for(int i = 0; i < plain.length; i++)
                plain[i] = (byte)(plain[i] ^ SECRET_KEY);
            stream.fill(plain);
        }
    }


    /**
     * This method determines how the internal state of the object will be
     * transcripted to a raw sequence of bytes.
     * @param b the <CODE>Stream</CODE> that will receive the data that
     * constitutes the current's object internal state.
     * @throws MarshalException if the serialization process cannot be
     * completed.
     */
    public void marshal(Stream b) throws MarshalException {
        b.write(this.SECRET_KEY);
    }

    /**
     * This method defines how the content of the key can be retrieved from a
     * raw sequence of bytes. 
     * @param b the <CODE>Stream</CODE> that will be read so that the new
     * content of the object can be obtained.
     * @throws MarshalException if the serialization process cannot be
     * completed.
     */
    public void unmarshal(Stream b) throws MarshalException {
        this.SECRET_KEY = b.readByte();
    }
}
