package arcademis.security;

import java.util.Date;

import arcademis.Stream;
import arcademis.Marshalable;
import arcademis.MarshalException;

/**
 * The signature has three components. A name, that allows the lookup agency to
 * find the public key of the client and a time stamp. The time stamp is
 * used in order to check the validity of the signature, and to avoid a
 * man-in-the-middle attack. After the discovery agency receives the
 * signed message, it sends a message back to the source, with a enciphered
 * version of the time stamp. If the source is able to replay, than it
 * confirns its identity. Notice that the time stamp is enciphered with the
 * source's public key, so, just it can replay the message.
 */
public class Signature implements Marshalable, java.io.Serializable {

    private String name = null;

    private Date timeStamp = null;

    public Signature() {
        this.name = "ANONYMOUS";
        this.timeStamp = new Date();
    }

    public Signature(String name) {
        this.name = name;
        this.timeStamp = new Date();
    }

    public String getName() {
        return this.name;
    }

    public Date getTimeStamp() {
        return this.timeStamp;
    }

    public void setName(String name) {
        this.name = name;
    }

    public void setTimeStamp(Date timeStamp) {
        this.timeStamp = timeStamp;
    }

    public String toString() {
        return "Digital certificate:\n" +
               "Owner: " + this.name + '\n' +
               "Time stamp: " + this.timeStamp;
    }

    public boolean equals(Object o) {
        if( !(o instanceof Signature) )
            return false;
        else {
            Signature sig = (Signature)o;
//            return sig.name.equals(this.name) && (sig.timeStamp.getTime() == this.timeStamp.getTime());
            return sig.name.equals(this.name);
        }
    }

    public int hashCode() {
        return this.name.hashCode();
    }

    /**
     * This method determines how the internal state of the object will be
     * transcripted to a raw sequence of bytes.
     * @param b the <CODE>Stream</CODE> that will receive the data that
     * constitutes the current's object internal state.
     * @throws MarshalException if the serialization process cannot be
     * completed.
     */
    public void marshal(Stream b) throws MarshalException {
        b.write(this.name);
        b.write(this.timeStamp.getTime());
    }

    /**
     * This method defines how the content of the key can be retrieved from a
     * raw sequence of bytes.
     * @param b the <CODE>Stream</CODE> that will be read so that the new
     * content of the object can be obtained.
     * @throws MarshalException if the serialization process cannot be
     * completed.
     */
    public void unmarshal(Stream b) throws MarshalException {
        this.name = (String)b.readObject();
        this.timeStamp = new Date(b.readLong());
    }

}
