package arcademis.security;

import arcademis.Stream;
import arcademis.MarshalException;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import java.math.BigInteger;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.interfaces.RSAKey;

import gnu.crypto.sig.rsa.EME_PKCS1_V1_5;
import gnu.crypto.sig.rsa.RSA;
import gnu.crypto.key.rsa.RSAKeyPairGenerator;
import gnu.crypto.key.rsa.GnuRSAPublicKey;

public class RsaKey implements Key, Serializable {

    private boolean isPublic = false;
    private RSAKey key = null;

    public RsaKey() {}

    public RsaKey(RSAKey key) {
        this.key = key;
        if(this.key instanceof RSAPublicKey)
            this.isPublic = true;
        else
            this.isPublic = false;
    }

    /**
     * This method encypher an array of bytes using the given key. For example,
     * in order to send a signed, secret message, the message can be signed with
     * the public key of the receiver, and the private key of the sender.
     * RSAES-PKCS1-V1_5-ENCRYPT ((n, e), M) (Section 7.2.1)
     * @param message the set of bytes that will be encrypted.
     * @return the encrypted message.
     */
    public byte[] encrypt(byte[] message) throws ArcademisCryptoException {
            if (!isPublic)
                    throw new ArcademisCryptoException("Key is not a RSAPublicKey");
            RSAPublicKey pkey = (RSAPublicKey) key;

            //EME-PKCS1-v1_5 encoding
            //EME_PKCS1_V1_5 eme = EME_PKCS1_V1_5.getInstance(key);
            //byte[] emsg = eme.encode (message);

            /**** RSA encryption ****/

            //Convert the encoded message EM to an integer message representative (OS2IP)
            //BigInteger m = new BigInteger (emsg);
            BigInteger m = new BigInteger (message);

           //RSAEP encryption
           //BigInteger c = RSA.encrypt(key, m);
           BigInteger c = RSA.encrypt(pkey, m);

           //Convert the ciphertext representative c to a ciphertext C of length k octets
//           byte[] cText = RSA.I2OSP (c, message.length);
           byte[] cText = c.toByteArray();

           return cText;
    }


    /**
     * This method is the inverse function of the <CODE>encryp</CODE> operation.
     * Given a cypher, and a key, it will apply the reverse function in order to
     * recover the original message.
     * RSAES-PKCS1-V1_5-DECRYPT (K, C) (Section 7.2.2)
     * @param message the set of bytes that will be decyphered.
     * @return the decrypted message.
     */
    public byte[] decrypt(byte[] message) throws ArcademisCryptoException {
            if (!isPublic)
                    throw new ArcademisCryptoException("Key is not a RSAPrivateKey");
            RSAPrivateKey pkey = (RSAPrivateKey) key;

            //we are not performing the length test.

            /**** RSA decryption ****/
            //Convert the ciphertext C to an integer ciphertext representative
            BigInteger c = new BigInteger (message);

            //RSADP decryption
            BigInteger iMsg = RSA.decrypt(pkey, c);

            //I2OSP
//            byte[] cText = RSA.I2OSP (iMsg, message.length);

            //EME-PKCS1-v1_5 decoding
            //EME_PKCS1_V1_5 eme = EME_PKCS1_V1_5.getInstance(key);
            //byte[] dmsg = eme.decode (message);

            byte[] dmsg = iMsg.toByteArray();
            return dmsg;
    }

    public static final int PAD_SIZE = 32;


    /**
     * This method receives a stream of bytes and sign it. The ciphered
     * data will replace the original data in the stream. The signature is
     * always performed with the private key. In order to recover the signed
     * data, it is necessary to use the <CODE>verify</CODE> method.
     * @param stream the object that holds the data to be encrypted. Notice
     * that the original data stored in this stream will be lost.
     * @throws ArcademisCryptoException if this is not a private key.
     */
    public void sign(Stream stream) throws ArcademisCryptoException {
        if(isPublic)
            throw new ArcademisCryptoException("Attempt to sign with " + this.key.getClass().getName());
        byte[] plain = stream.getBytes();
        int a = (int)(plain.length / PAD_SIZE);
        int b = (int)(plain.length % PAD_SIZE);
        int numBlocks = (b > 0) ? (a + 1) : a;
        try {
            stream.clean();
            stream.write(plain.length);
            stream.write(numBlocks);
            for(int i = 0; i < plain.length; ) {
                byte[] aux = null;
                if( (i + PAD_SIZE) < plain.length)
                    aux = new byte[PAD_SIZE];
                else
                    aux = new byte[plain.length - i];
                int j = 0;
                for(; j < aux.length; j++)
                    aux[j] = plain[j + i];
                BigInteger m = new BigInteger(aux);
                boolean isNegative = false;
                if(m.signum() < 0) {
                    m = m.negate();
                    isNegative = true;
                }
                BigInteger c = RSA.sign((RSAPrivateKey)key, m);
                byte[] cipher = c.toByteArray();
                stream.write(isNegative);
                stream.write(cipher);
                i = i + j;
            }
        } catch (MarshalException me) {
            me.printStackTrace();
        }
    }


    /**
     * This method verifiers a signed stream. To verify an stream means to
     * decipher with the public key the sequence of bytes that was encrypted
     * with the corresponding private key. The same stream given as
     * a parameter is used to hold the decrypted data. Only public keys
     * can be used to verify an stream.
     * @param stream the stream to be decrypted. The original contents of
     * this stream will be lost.
     * @throws ArcademisCryptoException if this is not a public key.
     */
    public void verify(Stream stream)  throws ArcademisCryptoException {
        if(!isPublic)
            throw new ArcademisCryptoException("Attempt to verify with " + this.key.getClass().getName());
        try {
            int size = stream.readInt();
            int numBlocks = stream.readInt();
            BigInteger[] m = new BigInteger[numBlocks];
            for(int i = 0; i < numBlocks; i++) {
                boolean isNegative = stream.readBoolean();
                byte[] cipher = (byte[])stream.readObject();
                BigInteger s = new BigInteger(cipher);
                BigInteger aux = RSA.verify((RSAPublicKey)key, s);
                m[i] = isNegative ? aux.negate() : aux;
            }
            byte[] plain = new byte[size];
            for(int i = 0, j = 0; i < numBlocks; i++) {
                byte aux[] = m[i].toByteArray();
                for(int k = 0; k < aux.length; k++)
                    plain[j++] = aux[k];
            }
            stream.clean();
            stream.fill(plain);
        } catch (MarshalException me) {
            me.printStackTrace();
        }
    }


    /**
     * This method receives a stream of bytes and encrypts it. The ciphered
     * data will replace the original data in the stream.
     * @param stream the object that holds the data to be encrypted. Notice
     * that the original data stored in this stream will be lost.
     */
    public void encrypt(Stream stream) throws ArcademisCryptoException {
        byte[] plain = stream.getBytes();
        int a = (int)(plain.length / PAD_SIZE);
        int b = (int)(plain.length % PAD_SIZE);
        int numBlocks = (b > 0) ? (a + 1) : a;
        try {
            stream.clean();
            stream.write(plain.length);
            stream.write(numBlocks);
            for(int i = 0; i < plain.length; ) {
                byte[] aux = null;
                if( (i + PAD_SIZE) < plain.length)
                    aux = new byte[PAD_SIZE];
                else
                    aux = new byte[plain.length - i];
                int j = 0;
                for(; j < aux.length; j++)
                    aux[j] = plain[j + i];
                BigInteger m = new BigInteger(aux);
                boolean isNegative = false;
                if(m.signum() < 0) {
                    m = m.negate();
                    isNegative = true;
                }
                if(!isPublic)
                    throw new ArcademisCryptoException("Attempt to encrypt with " + this.key.getClass().getName());
                BigInteger c = RSA.encrypt((RSAPublicKey)key, m);
                byte[] cipher = c.toByteArray();
                stream.write(isNegative);
                stream.write(cipher);
                i = i + j;
            }
        } catch (MarshalException me) {
            me.printStackTrace();
        }
    }


    /**
     * This method deciphers an encrypted stream. The same stream given as
     * a parameter is used to hold the decrypted data.
     * @param stream the stream to be decrypted. The original contents of
     * this stream will be lost.
     */
    public void decrypt(Stream stream)  throws ArcademisCryptoException {
        try {
            int size = stream.readInt();
            int numBlocks = stream.readInt();
            BigInteger[] m = new BigInteger[numBlocks];
            for(int i = 0; i < numBlocks; i++) {
                boolean isNegative = stream.readBoolean();
                byte[] cipher = (byte[])stream.readObject();
                BigInteger c = new BigInteger(cipher);
                if(isPublic)
                    throw new ArcademisCryptoException("Attempt to decrypt with " + this.key.getClass().getName());
                BigInteger aux = RSA.decrypt((RSAPrivateKey)key, c);
                m[i] = isNegative ? aux.negate() : aux;
            }
            byte[] plain = new byte[size];
            for(int i = 0, j = 0; i < numBlocks; i++) {
                byte aux[] = m[i].toByteArray();
                for(int k = 0; k < aux.length; k++)
                    plain[j++] = aux[k];
            }
            stream.clean();
            stream.fill(plain);
        } catch (MarshalException me) {
            me.printStackTrace();
        }
    }


    /**
     * This method determines how the internal state of the object will be
     * transcripted to a raw sequence of bytes.
     * @param b the <CODE>Stream</CODE> that will receive the data that
     * constitutes the current's object internal state.
     * @throws MarshalException if the serialization process cannot be
     * completed.
     */
    public void marshal(Stream b) throws MarshalException {
        if( !(this.key instanceof GnuRSAPublicKey) )
            throw new MarshalException("Error: public key is not instance of GnuRSAPublicKey");
        else {
            GnuRSAPublicKey gKey = (GnuRSAPublicKey)this.key;
            BigInteger n = gKey.getModulus();
            BigInteger e = gKey.getPublicExponent();
            b.write(n.toByteArray());
            b.write(e.toByteArray());
        }
    }

    /**
     * This method defines how the content of the key can be retrieved from a
     * raw sequence of bytes. 
     * @param b the <CODE>Stream</CODE> that will be read so that the new
     * content of the object can be obtained.
     * @throws MarshalException if the serialization process cannot be
     * completed.
     */
    public void unmarshal(Stream b) throws MarshalException {
        try {
            byte[] bn = (byte[])b.readObject();
            byte[] be = (byte[])b.readObject();
            BigInteger n = new BigInteger(bn);
            BigInteger e = new BigInteger(be);
            this.key = new GnuRSAPublicKey(n, e);
            this.isPublic = true;
        } catch (ClassCastException e) {
            throw new MarshalException("Error during public key serialization");
        }
    }
}
