package arcademis.security;

import arcademis.Stream;
import arcademis.Marshalable;

public interface Key extends Marshalable {
    /**
     * This method receives a stream of bytes and encrypts it. The ciphered
     * data will replace the original data in the stream.
     * @param stream the object that holds the data to be encrypted. Notice
     * that the original data stored in this stream will be lost.
     */
    public void encrypt(Stream stream) throws ArcademisCryptoException;

    /**
     * This method deciphers an encrypted stream. The same stream given as
     * a parameter is used to hold the decrypted data.
     * @param stream the stream to be decrypted. The original contents of 
     * this stream will be lost.
     */
    public void decrypt(Stream stream) throws ArcademisCryptoException;

    /**
     * This method receives a stream of bytes and sign it. The ciphered
     * data will replace the original data in the stream. The signature is
     * always performed with the private key. In order to recover the signed
     * data, it is necessary to use the <CODE>verify</CODE> method.
     * @param stream the object that holds the data to be encrypted. Notice
     * that the original data stored in this stream will be lost.
     * @throws ArcademisCryptoException if this is not a private key.
     */
    public void sign(Stream stream) throws ArcademisCryptoException;

    /**
     * This method verifiers a signed stream. To verify an stream means to
     * decipher with the public key the sequence of bytes that was encrypted
     * with the corresponding private key. The same stream given as
     * a parameter is used to hold the decrypted data. Only public keys
     * can be used to verify an stream.
     * @param stream the stream to be decrypted. The original contents of
     * this stream will be lost.
     * @throws ArcademisCryptoException if this is not a public key.
     */
    public void verify(Stream stream)  throws ArcademisCryptoException;
}
