package arcademis;

/**
 * This interface determines the <CODE>Invoker</CODE> factory. Whenever necessary to
 * modify the implementation of invokers, e.g. by adding new decorators to the
 * invoker chain it is sufficient to change the factory's implementation so it
 * will return the proper chain of invokers.
 * In order to add further functionalities to invokers it is possible to use
 * decorators. These are enclosing objects that intercept the <CODE>invoke</CODE>
 * method and perform some processing over its parameters. Chains of decorators
 * may be compoused in a more flexible way than if simple inherintance was used
 * instead. Such chains may be manipulated by the invoker factory.
 */
public interface InvokerFc {

	/**
	 * Creates a default invoker
	 * @return an object of the <CODE>Invoker</CODE> type.
	 */
	public Invoker createInvoker ();

	/**
	 * Creates a specific invoker
	 * @param objType defines the type of the invoker that will be created.
	 * @return an object of the <CODE>Invoker</CODE> type.
	 */
	public Invoker createInvoker (int objType);

	/**
	 * Adds a new decorator to the chain of decorators that will be generated by
	 * the factory.
	 * @param d an object of the <CODE>InvokerDecorator</CODE> type.
	 */
	public void insertDecorator(InvokerDecorator d);

	/**
	 * Remove all the decorators from the base component.
	 */
	public void removeDecorators();

	/**
	 * Defines the implementation of the base component.
	 * @param i the new base component.
	 */
	public void setBaseComponent(Invoker i);
}