package CFGV;

import java.util.HashSet;

import joeq.Compiler.Quad.*;
import joeq.Compiler.Quad.SSA.EnterSSA;
import joeq.Compiler.Dataflow.Solver;
import joeq.Compiler.Dataflow.UnionBitVectorFact;
import joeq.Compiler.Dataflow.LivenessAnalysis;
import joeq.Compiler.Dataflow.IterativeSolver;

import joeq.Class.jq_Class;
import joeq.Class.jq_ClassInitializer;
import joeq.Class.jq_InstanceMethod;
import joeq.Class.jq_Initializer;
import joeq.Class.jq_Method;
import joeq.Class.jq_MethodVisitor;
import joeq.Class.jq_StaticMethod;

import joeq.Main.Helper;

import joeq.Util.Templates.List;
import joeq.Util.Templates.ListIterator;

/**
 * This class is implements a visitor of methods that prints the control flow
 * data of each method visited. Such information will be printed in a series of
 * .dot files, each file starting with the name of the target class, and ending
 * with the method name that is described there.
 */
public class DrawerVisitor implements jq_MethodVisitor {

    /**
     * This table contains the blocks that describe branches. Those are special
     * blocks because they can have more than one successor, they just have one
     * instruction, and they are printed as diamonds.
     */
    private HashSet ifBlocks = null;

    /**
     * The name of the class been analyzed. One .dot file will be created for
     * each method of the class.
     */
    private String className = null;

    /**
     * The name of the directory where the generated files must be put.
     */
    private String directoryName = null;

    /**
     * This string is used to parse different command line arguments.
     */
    private int option = 0;

    /**
     * Constructor method.
     */
    public DrawerVisitor() {
        this.ifBlocks = new HashSet();
        this.directoryName = "";
        this.option = 0;
    }


    /**
     * This method defines how the data must be generated.
     * @param option an integer number that describes how data should be
     * generated. Possibilities are:
     * <PRE>
     * option = 0: the plain control flow graph.
     * option &amp; 1 == 1: program is converted to SSA.
     * option &amp; 2 == 2: the if blocks are created.
     * option &amp; 4 == 4: the phi functions are destroyed and substituted by
     * move instructions.
     * </PRE>
     * By using different bit masks, differents options can be used.
     */
    public void setOption(int option) {
        this.option = option;
    }


    /**
     * Determines the name of the directory in which the generated files must
     * be put.
     * @param directoryName the name of the directory.
     */
    public void setDirectoryName(String directoryName) {
        this.directoryName = directoryName;
    }


    /**
     * Defines the name of the class that will give the methods to be drawn.
     * the name of the class will be part of the name of the generated files.
     * @param className the name of the class.
     */
    public void setClassName(String className) {
        this.className = className;
    }

    /**
     * This visitor traverses the constructor methods.
     * @param m the descriptor of the contructor method.
     */
    public void visitClassInitializer(jq_ClassInitializer m) {}

    /**
     * This visitor traverses the contructor methods.
     * @param m the descriptor of the contructor method.
     */
    public void visitInitializer(jq_Initializer m) {}

    /**
     * This visitor traverses the instance methods. Instance methods are not
     * static.
     * @param m the instance method.
     */
    public void visitInstanceMethod(jq_InstanceMethod m) {}

    /**
     * This visitor traverses the static methods.
     * @param m a static method.
     */
    public void visitStaticMethod(jq_StaticMethod m) {}

    /**
     * This visitor traverses all the methods in the class. Both static and
     * non-static methods are visited.
     * @param m a java method.
     */
    public void visitMethod(jq_Method m) {
        SimpleGraphDrawer dp = new SimpleGraphDrawer();
        Helper.runPass(m, dp);
    }


    /**
     * This method converts the encapsulated graph to SSA representation.
     * In this representation, every variable is defined just once.
     * @param cfg the control flow of the graph.
     */
    public static void convertSSA(ControlFlowGraph cfg) {
        EnterSSA.markSSARegisterFlags(cfg);
        EnterSSA essa = new EnterSSA();
        essa.visitCFG(cfg);
    }


    /**
     * In JoeQ, the branches come in the end of the blocks. This method creates
     * a separate method for each branch.
     * @param cfg the control flow of the graph.
     * @param bb the basic block to be analyzed.
     */
    private void splitIfBlocks(ControlFlowGraph cfg, BasicBlock bb) {
        Quad lastQuad = bb.getLastQuad();
        if(lastQuad != null) {
            Operator op = lastQuad.getOperator();
            if(op instanceof Operator.Branch) {
                // 1) creates a new block
                BasicBlock newBlock = cfg.createBasicBlock(1, bb.getNumberOfSuccessors(), 1, null);
                // 2) add the last quad to the new block
                newBlock.appendQuad(lastQuad);
                // 3) removes the last quad from the last block
                bb.removeQuad(lastQuad);
                // 4) make the successors of the old block the successors of the new one
                List.BasicBlock succList = bb.getSuccessors();
                ListIterator.BasicBlock succs = succList.basicBlockIterator();
                while(succs.hasNext()) {
                    BasicBlock succ = (BasicBlock)succs.next();
                    newBlock.addSuccessor(succ);
                    succ.addPredecessor(newBlock);
                    // 5) remove the old block as predecessor of its successors
                    succ.removePredecessor(bb);
                }
                // 6) remove the successors of the old block
                bb.removeAllSuccessors();
                if(bb.size() > 0 || bb.isEntry()) {
                    // 7) make the new block the successor of the old one
                    bb.addSuccessor(newBlock);
                    // 8) make the old block the predecessor of the new one
                    newBlock.addPredecessor(bb);
                    // 9) mark as an if block, to be rendered properly
                } else {
                    // the old basic block had just one quad, and it was a branch
                    List.BasicBlock predList = bb.getPredecessors();
                    ListIterator.BasicBlock preds = predList.basicBlockIterator();
                    while(preds.hasNext()) {
                        BasicBlock pred = (BasicBlock)preds.next();
                        newBlock.addPredecessor(pred);
                        pred.addSuccessor(newBlock);
                        pred.removeSuccessor(bb);
                    }
                }
                ifBlocks.add(newBlock);
            }
        }
    }


    /**
     * This class implements a control flow graph visitor. Such visitor
     * traverses all the nodes of a control flow object, and generates a .dot
     * file that contains a graphic representation of the control flow.
     */
    private class SimpleGraphDrawer implements ControlFlowGraphVisitor {

        /**
         * This method implemets the actions that must be taken when the
         * control flow graph is been visited.
         */
        public void visitCFG(ControlFlowGraph cfg) {

            if((option & 1) != 0) {
                DrawerVisitor.convertSSA(cfg);
            }
            if((option & 2) != 0) {
                ListIterator.BasicBlock bbIterator = cfg.postOrderOnReverseGraphIterator();
                while(bbIterator.hasNext()) {
                    BasicBlock bb = bbIterator.nextBasicBlock();
                    splitIfBlocks(cfg, bb);
                }
            }
            if((option & 4) != 0) {
                PhiDestructor sculp = new PhiDestructor(cfg);
                sculp.destroyPhiFunctions();
            }

            String methodName = cfg.getMethod().getName().toString();
            CFG cfgRenderer = new CFG(className, methodName);

            Solver solver = this.getLivenessAnalysis(cfg);

            ListIterator.BasicBlock bbIterator = cfg.postOrderOnReverseGraphIterator();
            while(bbIterator.hasNext()) {
                BasicBlock bb = bbIterator.nextBasicBlock();
                // collects liveness analysis information
                UnionBitVectorFact f = (UnionBitVectorFact)solver.getDataflowValue(bb);
                jwutil.math.BitString bs = f.getBitString(); 
                if(ifBlocks.contains(bb))
                    cfgRenderer.add(new BranchBlockWrapper(bb), bs);
                else
                    cfgRenderer.add(new BasicBlockWrapper(bb), bs);
            }

            String fileName = directoryName + '/' + className + '.' + methodName + '.' + "dot";
            System.err.println("Rendering " + fileName + " ...");
            cfgRenderer.print(fileName);
        }


        /**
         * Performs liveness analysis in the control flow graph.
         * @param cfg the control flow to be analyzed.
         * @return the solution of the liveness analyses.
         */
        public Solver getLivenessAnalysis(ControlFlowGraph cfg) {
            LivenessAnalysis la = new LivenessAnalysis();
            Solver solver = new IterativeSolver();
            jwutil.graphs.EdgeGraph eg = new jwutil.graphs.EdgeGraph(new jwutil.graphs.ReverseGraph(cfg, java.util.Collections.singleton(cfg.exit())));
            solver.initialize(la, eg);
            solver.solve();
            return solver;
        }
    }


    public static void main(String args[]) {
        if(args.length == 0) {
            System.out.println("Sintaxe: java DrawerVisitor class [args]");
            System.out.println("-d <dir> Destination directory for output files.");
            System.out.println("-s       convert the code to SSA.");
            System.out.println("-p       creates a separete block for branch instructions");
            System.out.println("-x       destroes PHI functions. In this case, use also -s -p");
            System.exit(0);
        } else {
            DrawerVisitor dv = new DrawerVisitor();
            try {
                // gets the class' name:
                jq_Class cl = (jq_Class)Helper.load(args[0]);
                jq_InstanceMethod[] mList = cl.getDeclaredInstanceMethods();

                // parse the command line arguments:
                int option = 0;
                for (int i = 1; i < args.length; i++) {
                    if (args[i].equals("-d"))
                        if(args.length > (i+1))
                            dv.setDirectoryName(args[++i]);
                        else {
                            System.err.println("Error: -d without directory name");
                            System.exit(1);
                        }
                    else if (args[i].equals("-s"))
                        option += 1;
                    else if (args[i].equals("-p"))
                        option += 2;
                    else if (args[i].equals("-x"))
                        option += 4;
                    else if (args[i].startsWith("-")) {
                        System.out.println("Unknown option: "+args[i]);
                        System.exit(-1);
                    }
                }
                dv.setOption(option);

                // generate the .dot files:
                for(int j = 0; j < mList.length; j++) {
                    dv.setClassName(cl.getName());
                    Helper.runPass(mList[j], dv);
                }
            } catch (Exception e) {
                System.err.println("Error in: " + args[0]);
                e.printStackTrace();
            }
        }
    }

}

/*
                jq_Class cl = (jq_Class)Helper.load(args[0]);
                jq_InstanceMethod[] mList = cl.getDeclaredInstanceMethods();
                if(args.length > 1)
                    dv.setDirectoryName(args[1]);
                dv.setOption(7);
                for(int j = 0; j < mList.length; j++) {
                    dv.setClassName(cl.getName());
                    Helper.runPass(mList[j], dv);
                }
 */
