package CFGV;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.ListIterator;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.FileWriter;
import java.io.BufferedWriter;
import joeq.Compiler.Quad.BasicBlock;
import jwutil.math.BitString;

/**
 * This class is used to generate a .dot file containing a representation of
 * the code from one java method.
 */
public class CFG {

    /**
     * The name of the java class that is been analyzed.
     */
    private String className = null;

    /**
     * The name of the method that is the source of the code that will
     * constitute the control flow to be drawn.
     */
    private String methodName = null;

    /**
     * the list of basic blocks to be drawn. During the drawing phase, this
     * list is scanned and each basic block is printed.
     */
    private LinkedList list = null;


    /**
     * this hash table binds basic blocks to bit vectors that describe which
     * variables are alive in the entrance of such basic blocks.
     */
    private HashMap livenessInf = null;


    /**
     * Constructor method.
     * @param className the name of the class whose method will be analyzed.
     * @param methodName the name of the method that will be analyzed.
     */
    public CFG(String className, String methodName) {
        this.className = className;
        this.methodName = methodName;
        this.list = new LinkedList();
        this.livenessInf = new HashMap();
    }


    /**
     * This method adds a basic block to the list of basic blocks that will be
     * drawn.
     * @param n the basic block that will be drawn.
     * @param bs a bit vector that describes the set of temporaries that are
     * alive in the entrance of the basic block.
     */
    public void add(CfgNode n, BitString bs) {
        list.add(n);
        livenessInf.put(n, bs);
    }


    /**
     * This method defines the header of a .dot file. It will return a string
     * giving a short
     * comment about the nature of the graph been described, such as the name
     * of the class and the name of the method. The string will contain a
     * definition of nodes in the graph, e.g.: font size, type of border, etc.
     * @return an object of the String type.
     */
    private String getHeaderInformation() {
        String header = "";

        header += "/*\n";
        header += " * This is a representation of the control flow graph of\n";
        header += " * the method " + this.methodName + '\n';
        header += " * from the class " + this.className + '\n';
        header += " */\n";
        header += "digraph " + this.methodName + " {\n\n";
        header += "node [\n";
        header += "    fontname = \"Arial\"\n";
        header += "    fontsize = \"10\"\n";
        header += "    shape = \"box\"\n";
        header += "];\n\n";
        header += "edge [\n";
        header += "    fontname = \"Arial\"\n";
        header += "    fontsize = \"10\"\n";
        header += "    shape = \"box\"\n";
        header += "];\n\n";

        return header;
    }


    /**
     * This method generates a .dot file describing the control flow of a java
     * method.
     * @param fileName the name of the .dot file to be generated.
     */
    public void print(String fileName) {
        try {
            PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
            out.println(this.getHeaderInformation());
            ListIterator li = list.listIterator();
            while(li.hasNext()) {
                CfgNode n = (CfgNode)li.next();
                n.printNode(out);
            }
            li = list.listIterator();
            while(li.hasNext()) {
                CfgNode n = (CfgNode)li.next();
                n.printEdges(out, (BitString)livenessInf.get(n));
            }
            out.println("}");
            out.close();
        } catch (java.io.IOException ioe) {
            ioe.printStackTrace();
        }
    }
}
