package CFGV;

import java.util.LinkedList;
import joeq.Compiler.Quad.*;
import joeq.Util.Templates.*;
import java.io.PrintWriter;
import java.io.PrintStream;
import jwutil.math.BitString;

/**
 * This class represents a branch in the control flow graph. This block should
 * contain just one instruction, which can be an if, a go-to or a switch.
 * Thus, even though the block contains just one instruction, it can contain
 * more than one successor. If it is a switch, it can contain several
 * successors.
 */
public class BranchBlockWrapper implements CfgNode {

    /**
     * This is the basic block that will be drawn in .dot format.
     */
    private BasicBlock bb = null;


    /**
     * Constructor method.
     * @param bb the basic block that will be represented by this graphic
     * block. The basic block contains java instructions.
     */
    public BranchBlockWrapper(BasicBlock bb) {
        this.bb = bb;
    }


    /**
     * This method prints code for a branch block in the given file.
     * @param out the output device.
     */
    public void printNode(PrintStream out) {
        out.println("\"" + this.bb.getID() + "\" [");
        out.println("    shape = \"diamond\"");
        out.println(this.getLabel());
        out.println("];\n");
    }


    /**
     * This method prints code for a block in the given file. The block will be
     * drawn as a diamond.
     * @param out the output file.
     */
    public void printNode(PrintWriter out) {
        out.println("\"" + this.bb.getID() + "\" [");
        out.println("    shape = \"diamond\"");
        out.println(this.getLabel());
        out.println("];\n");
    }


    /**
     * This method prints code for all the edges that are reaching this node.
     * The label of the edges contains the liveness information of the basic
     * block.
     * @param out the output device.
     * @param bs a vector containing liveness information.
     */
    public void printEdges(PrintStream out, BitString bs) {
        String label = BasicBlockWrapper.convertBitString(bs);
        List.BasicBlock lbb = this.bb.getPredecessors();
        for(int i = 0; i < lbb.size(); i++) {
            BasicBlock bbAux = lbb.getBasicBlock(i);
            out.println("\n\"" + bbAux.getID() + '\"' + " -> " + '\"' + this.bb.getID() + "\" [");
            out.println("    label=\"" + label + "\"");
            out.println("];");
        }
    }


    /**
     * This method prints code for all the edges that are reaching this node in
     * the given file.
     * The label of the edges contains the liveness information of the basic
     * block.
     * @param out the output file.
     * @param bs a vector containing liveness information.
     */
    public void printEdges(PrintWriter out, BitString bs) {
        String label = BasicBlockWrapper.convertBitString(bs);
        List.BasicBlock lbb = this.bb.getPredecessors();
        for(int i = 0; i < lbb.size(); i++) {
            BasicBlock bbAux = lbb.getBasicBlock(i);
            out.println("\n\"" + bbAux.getID() + '\"' + " -> " + '\"' + this.bb.getID() + "\" [");
            out.println("    label=\"" + label + "\"");
            out.println("];");
        }
    }


    /**
     * This method prints the label of a basic block. The label contains the
     * block identifier, e.g.: <CODE>Block 1</CODE>, and the set of
     * instructions that are present in the Java program. Because the block is
     * a diamond, and has just one instruction, it is more compact that the
     * analougous method in the basic block class.
     * @return a string object
     */
    public String getLabel() {
        String label = "    label = \"Block " + bb.getID() + "\\n";
        ListIterator.Quad ql = this.bb.iterator();
        while(ql.hasNext()) {
            Quad q = ql.nextQuad();
            label += this.quadToText(q) + "\\n";
        }
        label += "\"";
        return label;
    }


    /**
     * This method generates a textual representation of the quad. This
     * representation contains the name of the quad, its number, the set of
     * used registers, and the set of defined registers. The output is more
     * compact than the analogous code in the basic block class.
     * @param q the quad to be parsed
     * @return a string object
     */
    public String quadToText(Quad q) {
        String defs = " ";
        List.RegisterOperand l = q.getDefinedRegisters();
        for(int i = 0; i < l.size(); i++) {
            defs += l.getRegisterOperand(i).getRegister().toString() + "   ";
        }
        defs += ' ';

        String uses = " ";
        l = q.getUsedRegisters();
        for(int i = 0; i < l.size(); i++) {
            uses += l.getRegisterOperand(i).getRegister().toString() + " ";
        }
 
        return q.toString_short() + " " + defs + " <= " + uses;
    }

}
