package CFGV;

import java.util.LinkedList;
import java.util.StringTokenizer;
import joeq.Compiler.Quad.*;
import joeq.Util.Templates.*;
import java.io.PrintWriter;
import java.io.PrintStream;
import jwutil.math.BitString;

/**
 * This class represents a basic block in the control flow graph. A basic block
 * does not contain branches, that is, it just have one successor.
 */
public class BasicBlockWrapper implements CfgNode {

    /**
     * This is the basic block that will be drawn in .dot format.
     */
    private BasicBlock bb = null;

    /**
     * Constructor method.
     * @param bb the basic block that will be represented by this graphic
     * block. The basic block contains java instructions.
     */
    public BasicBlockWrapper(BasicBlock bb) {
        this.bb = bb;
    }

    /**
     * This method prints code for a block in the given file.
     * @param out the output file.
     */
    public void printNode(PrintStream out) {
        out.println("\"" + this.bb.getID() + "\" [");
        out.println(this.getLabel());
        out.println("];\n");
    }


    /**
     * This method prints code for a block in the given file.
     * @param out the output file.
     */
    public void printNode(PrintWriter out) {
        out.println("\"" + this.bb.getID() + "\" [");
        out.println(this.getLabel());
        out.println("];\n");
    }


    /**
     * This method prints code for all the edges that are reaching this node.
     * The label of the edges contains the liveness information of the basic
     * block.
     * @param out the output device.
     * @param bs a vector containing liveness information.
     */
    public void printEdges(PrintStream out, BitString bs) {
        String label = BasicBlockWrapper.convertBitString(bs);
        List.BasicBlock lbb = this.bb.getPredecessors();
        for(int i = 0; i < lbb.size(); i++) {
            BasicBlock bbAux = lbb.getBasicBlock(i);
            out.println("\n\"" + bbAux.getID() + '\"' + " -> " + '\"' + this.bb.getID() + "\" [");
            out.println("    label=\"" + label + "\"");
            out.println("];");
        }
    }


    /**
     * This method prints code for all the edges that are reaching this node in
     * the given file.
     * The label of the edges contains the liveness information of the basic
     * block.
     * @param out the output file.
     * @param bs a vector containing liveness information.
     */
    public void printEdges(PrintWriter out, BitString bs) {
        String label = BasicBlockWrapper.convertBitString(bs);
        List.BasicBlock lbb = this.bb.getPredecessors();
        for(int i = 0; i < lbb.size(); i++) {
            BasicBlock bbAux = lbb.getBasicBlock(i);
            out.println("\n\"" + bbAux.getID() + '\"' + " -> " + '\"' + this.bb.getID() + "\" [");
            out.println("    label=\"" + label + "\"");
            out.println("];");
        }
    }


    /**
     * This method prints the label of a basic block. The label contains the
     * block identifier, e.g.: <CODE>Block 1</CODE>, and the set of
     * instructions that are present in the Java program.
     * @return a string object
     */
    private String getLabel() {
        String label = "    label = \"Block " + bb.getID() + "\\l";
        ListIterator.Quad ql = this.bb.iterator();
        while(ql.hasNext()) {
            Quad q = ql.nextQuad();
            label += this.quadToText(q) + "\\l";
        }
        label += "\"";
        return label;
    }


    /**
     * This method generates a textual representation of the quad. This
     * representation contains the name of the quad, its number, the set of
     * used registers, and the set of defined registers.
     * @param q the quad to be parsed
     * @return a string object
     */
    public String quadToText(Quad q) {
        String defs = "    ";
        List.RegisterOperand l = q.getDefinedRegisters();
        for(int i = 0; i < l.size(); i++) {
            defs += l.getRegisterOperand(i).getRegister().toString() + "   ";
        }
        defs += ' ';

        String uses = "   ";
        l = q.getUsedRegisters();
        for(int i = 0; i < l.size(); i++) {
            uses += l.getRegisterOperand(i).getRegister().toString() + "  ";
        }
 
        return q.toString_short() + "  " + defs + "<= " + uses;
    }


    /**
     * This method gives the label of an edge. The label contains the set of
     * variables that are alive in the entry of the basic block.
     * @param bs a bit string containing the liveness information.
     * @return a string object.
     */
    public static String convertBitString(BitString bs) {
        String ans = "";
        String line = bs.toString();
        StringTokenizer st = new StringTokenizer(line, "{}, ");
        while(st.hasMoreTokens()) {
            String tk = st.nextToken();
            int reg = Integer.parseInt(tk);
            ans += "R" + (reg-1) + "  ";
        }
        return ans;
    }

}
